function [U,s,V] = bsvd(B_k)
%BSVD SVD of a bidiagonal matrix stored in "compact form".
%
% s = bsvd(B_k)
% [U,s,V] = bsvd(B_k)
%
% Computes the singular values, or the compact SVD, of the
% bidiagonal matrix B stored in compact form in B_k.
%
% If the bottom right element of B_k is a NAN, then B_k repre-
% sents an upper bidiagonal matrix (such as produced by bidiag),
% stored with its diagonal and upper bidiagonal in the first and
% second columns of B_k, repsectively.
%
% Otherwise, B_k represents a lower bidiagonal matrix (such as
% produced by lanc_b), stored with its lower bidiagonal and its
% diagonal in the first and second columns of B_k, respectively.

% Per Christian Hansen, UNI-C, 03/11/92.

% Initialization.
[k,l] = size(B_k);
if (l~=2), error('B_k does not represent a bidiagonal matrix'), end

% Determine which bidiagonal form.
if (B_k(k,2)==NaN)
  B = diag(B_k(:,1)) + diag(B_k(1:k-1,2),1);
else
  B = diag(B_k(:,1),-1) + diag([B_k(:,2);0]);
  [k1,k1] = size(B); B = B(:,1:k1-1);
end

% Compute the SVD.
if (nargout<=1)
  U = svd(B);
else
  [U,s,V] = csvd(B);
end
