function x = gen_form(L_p,x_s,A,b,K,M)
%GEN_FORM Transform a standard-form problem back to the general-form setting.
%
% x = gen_form(L_p,x_s,A,b,K,M)    (method 1)
% x = gen_form(L_p,x_s,x_0)        (method 2)
%
% Transforms the standard-form solution x_s back to the required
% solution to the general-form problem:
%    x = L_p*x_s + d ,
% where L_p and d depend on the method as follows:
%    method = 1: L_p = pseudoinverse of L, d  = K*(b - A*L_p*x_s)
%    method = 2: L_p = A-weighted pseudoinverse of L, d = x_0.
%
% Usually, the standard-form problem is generated by means of
% function std_form.
%
% Note that x_s may have more that one column.

% References: L. Elden, "Algorithms for regularization of ill-
% conditioned least-squares problems", BIT 17 (1977), 134-145.
% L. Elden, "A weighted pseudoinverse, generalized singular values,
% and constrained lest squares problems", BIT 22 (1982), 487-502.
% M. Hanke, "Regularization with differential operators.  An itera-
% tive approach", J. Numer. Funct. Anal. Optim. 13 (1992), 523-540.

% Per Christian Hansen, UNI-C, 06/12/93.

% Nargin determines which method.
if (nargin==6)
  [p,q] = size(x_s); [Km,Kn] = size(K);
  if (Km==0)
    x = L_p*x_s;
  else
    x = L_p*x_s + K*(M*(b*ones(1,q) - A*(L_p*x_s)));
  end
else
  x_0 = A; [p,q] = size(x_s);
  x = L_p*x_s + x_0*ones(1,q);
end
