/* lpsolve.c

  interface to the lp_solve 5.1 toolkit.

  This file and the MATLAB interface code is originally developed by
  Jeffrey C. Kantor (jeff@control.cheg.nd.edu, cchen@darwin.helion.nd.edu)
  for lp_solve version 2.3
  The original name was lpmex.c


  It is completely revised and redesigned by
  Peter Notebaert (peno@mailme.org)
  for lp_solve version 5.1


  lpsolve.c needs to be linked with hash.c and the lp_solve library.


  The hash function is needed to have a quick access from the provided functionname to the
  implementation of the function.

  Peter Notebaert

*/

#define driverVERSION "5.1.0.0"

#include <stdio.h>
#include <string.h>

#include "lpsolvecaller.h"

/* Declare a global lp record */

#define LPSTEP 100

static lprec   **lp;
static lprec   *lp0;
static hashtable *hashtab;
static int     h;
static int     lp_last;
static int     result;

/* Some working storage */

static short   initialized = FALSE;
static char    *cmd; /* command */
static char    *errmsg; /* error message */

static double  *pr;

#define buf    errmsg

#define cmdsz     50
#define errmsgsz 200
#define bufsz    errmsgsz

typedef void (impl_routine)();

static void Check_nrhs(char *functionname, int nrhs0, int nrhs);
static void impl_set_obj_fn();

static double *PrepareCall(char *functionname, int nrhs0, int nrhs, pMatrix plhs)
{
	Check_nrhs(functionname, nrhs0, nrhs);
        return((plhs == NULL) ? NULL : CreateDoubleMatrix(1, 1, plhs));
}

static void Check_nrhs(char *functionname, int nrhs0, int nrhs)
{
	if (nrhs - 1 != nrhs0) {
                sprintf(errmsg, "%s requires %d argument%s.", functionname, nrhs0, (nrhs0 == 1) ? "" : "s");
		ErrMsgTxt(errmsg);
	}
}


/* callback function for lp_solve. Messages are reported via this routine and printed in the application */

static void __WINAPI mylog(lprec *lp, void *userhandle, char *buf)
{
  	Printf("%s", buf);
}


/* put lp_handle on list */

static int create_handle(lprec *lp0, char *err)
{
        int i;

        if (lp0 == NULL)
        	ErrMsgTxt(err);
	for (i = 0; (i <= lp_last) && (lp[i] != NULL); i++);
	if (i > lp_last) {
	  	i = ++lp_last;
                if ((i % LPSTEP) == 0) {
                        if (i == 0)
                                lp = (lprec **) malloc(LPSTEP * sizeof(*lp));
                        else
                                lp = (lprec **) realloc(lp, (i + LPSTEP) * sizeof(*lp));
                        memset(lp + i, 0, LPSTEP * sizeof(*lp));
                }
        }
        lp[i] = lp0;

        putlogfunc(lp0, mylog, NULL);

        return(i);
}

/* lp_handle validation test */

static int handle_valid(int handle)
{
        if (handle < 0 || handle > lp_last || lp[handle] == NULL)
	   return(0);
        else
           return(1);
}

/* free lp_handle from list */

static void delete_handle(int handle)
{
        if (handle_valid(handle)) {
                lprec *lp0 = lp[handle];

                delete_lp(lp0);
                lp[handle] = NULL;
        }
}


/* An exit function to clean up the lp structure.
   called on exit */

void ExitFcn()
{
	int	i;

        if (initialized) {
        	for (i = 0; i <= lp_last; i++)
                        delete_handle(i);
                free_hash_table(hashtab);
                free(cmd);
        	free(errmsg);
                exit_lpsolve_lib();
#               if defined DEBUG
		        Printf("Terminated\n");
#               endif
        }
}


#if defined DEBUG || defined DEMO
/* xxlpsolve('demo') */
/* This demo is not a necessary part of the program */

static void impl_demo()
{
        int i, h;

	/* Printf("%.15f", GetRealScalar(prhs + 1)); */

        PrepareCall(cmd, 0, nrhs, NULL);

        h = create_handle(make_lp(0, 4), "make_lp failed");
        lp0 = lp[h];

        Printf("min: 2 C1 +3 C2 -2 C3 +3 C4;\n");
	str_set_obj_fn(lp0, "2 3 -2 3");

        Printf("3 C1 +2 C2 +2 C3 +1 C4 <= 4.0;\n");
        str_add_constraint(lp0, "3 2 2 1", LE, 4.0);

        Printf("0 C1 +4 C2 +3 C3 +1 C4 >= 3.0;\n");
	str_add_constraint(lp0, "0 4 3 1", GE, 3.0);

        Printf("solve: %d\n", solve(lp0));
        Printf("obj: %f\n", get_objective(lp0));
        for (i = 1; i <= get_Ncolumns(lp0); i++)
        	Printf("C%d: %f\n", i, get_var_primalresult(lp0, get_Nrows(lp0) + i));
        write_lp(lp0, "a.lp");

        delete_handle(h);
}
#endif


/* since always a matrix vector is provided to both add_column and add_columnex, always call the more
   performant sparse version of the two routines */
/* return = xxlpsolve('add_column', lp_handle, [column]) */
/* return = xxlpsolve('add_columnex', lp_handle, [column]) */

static void impl_add_column()
{
        int m, count;
        int	*index;
	REAL	*vec;

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        m = get_Nrows(lp0);
        vec = (REAL *) matCalloc(1 + m, sizeof(*vec));
        index = (int *) matCalloc(1 + m, sizeof(*index));
	count = GetRealSparseVector(prhs + 2, vec, index, 0, 1 + m, 0);
	result = add_columnex(lp0, count, vec, index);
        *pr = (double) result;
        matFree(index);
        matFree(vec);
}


/* since always a matrix vector is provided to both add_constraint and add_constraintex, always call the more
   performant sparse version of the two routines */
/* return = xxlpsolve('add_constraint', lp_handle, [row], constr_type, rh) */
/* return = xxlpsolve('add_constraintex', lp_handle, [row], constr_type, rh) */

static void impl_add_constraint()
{
        int type, n, count;
        int	*index;
	REAL	*vec, value;

        pr = PrepareCall(cmd, 4, nrhs, plhs);
	type = (int) GetRealScalar(prhs + 3);
	if ((type != LE) && (type != EQ) && (type != GE)) {
		ErrMsgTxt("invalid constraint type.");
	}
        value = GetRealScalar(prhs + 4);
        n = get_Ncolumns(lp0);
        vec = (REAL *) matCalloc(n, sizeof(*vec));
        index = (int *) matCalloc(n, sizeof(*index));
	count = GetRealSparseVector(prhs + 2, vec, index, 1, n, 0);
	result = add_constraintex(lp0, count, vec, index, type, value);
        *pr = (double) result;
        matFree(index);
        matFree(vec);
}


/* return = xxlpsolve('add_SOS', lp_handle, name, sostype, priority, [sosvars], [weights]) */

static void impl_add_SOS()
{
        int n, *sosvars;
        REAL *weights;
        int count1, count2;

        pr = PrepareCall(cmd, 6, nrhs, plhs);
        GetString(prhs + 2, buf, bufsz);
        n = get_Ncolumns(lp0);
        sosvars = (int *) matCalloc(n, sizeof(*sosvars));
        weights = (REAL *) matCalloc(n, sizeof(*weights));
	count1 = GetIntVector(prhs + 5, sosvars, 0, n, FALSE);
        count2 = GetRealVector(prhs + 6, weights, 0, n, FALSE);
        if (count1 != count2) {
          matFree(weights);
          matFree(sosvars);
          ErrMsgTxt("add_SOS: sosvars and weights vector must have same size.");
        }
	result = add_SOS(lp0, buf, (int) GetRealScalar(prhs + 3), (int) GetRealScalar(prhs + 4), count1, sosvars,weights);
        *pr = (double) result;
        matFree(weights);
        matFree(sosvars);
}


/* return = xxlpsolve('column_in_lp', lp_handle, [column]) */

static void impl_column_in_lp()
{
        int n;
	REAL	*vec;

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        n = get_Nrows(lp0);
        vec = (REAL *) matCalloc(1 + n, sizeof(*vec));
        GetRealVector(prhs + 2, vec, 0, 1 + n, TRUE);
        *pr = (double) column_in_lp(lp0, vec);
        matFree(vec);
}


/* xxlpsolve('default_basis', lp_handle) */

static void impl_default_basis()
{
        PrepareCall(cmd, 1, nrhs, NULL);
        default_basis(lp0);
}


/* return = xxlpsolve('del_column', lp_handle, column) */

static void impl_del_column()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) del_column(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('del_constraint', lp_handle, del_row) */

static void impl_del_constraint()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) del_constraint(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('delete_lp', lp_handle) */
/* xxlpsolve('free_lp', lp_handle) */

static void impl_delete_lp()
{
        PrepareCall(cmd, 1, nrhs, NULL);
        delete_handle(h);
}


/* return = xxlpsolve('get_anti_degen', lp_handle) */
static void impl_get_anti_degen()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_anti_degen(lp0);
}


/* [bascolumn] = xxlpsolve('get_basis', lp_handle {, nonbasic}) */

static void impl_get_basis()
{
        int n, i, *bascolumn, *bascolumn0;
        MYBOOL nonbasic;

        if (nrhs == 1+1)
                n = 1;
        else
                n = 2;
        PrepareCall(cmd, n, nrhs, NULL);
        if (n == 1)
                nonbasic = 0;
        else
        	nonbasic = (MYBOOL) GetRealScalar(prhs + 2);
        n = get_Nrows(lp0) + ((nonbasic) ? get_Ncolumns(lp0) : 0);
        bascolumn0 = bascolumn = (int *) matCalloc(1 + n, sizeof(*bascolumn));
        get_basis(lp0, bascolumn, nonbasic);
        pr = CreateDoubleMatrix(n, 1, plhs);
        for (i = 0; i < n; i++)
          *(pr++) = *(++bascolumn);
        matFree(bascolumn0);
}


/* return = xxlpsolve('get_basiscrash', lp_handle) */

static void impl_get_basiscrash()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_basiscrash(lp0);
}


/* return = xxlpsolve('get_bb_depthlimit', lp_handle) */

static void impl_get_bb_depthlimit()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_bb_depthlimit(lp0);
}


/* return = xxlpsolve('get_bb_floorfirst', lp_handle) */

static void impl_get_bb_floorfirst()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_bb_floorfirst(lp0);
}


/* return = xxlpsolve('get_bb_rule', lp_handle) */

static void impl_get_bb_rule()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_bb_rule(lp0);
}


/* return = xxlpsolve('get_bounds_tighter', lp_handle) */

static void impl_get_bounds_tighter()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_bounds_tighter(lp0);
}


/* return = xxlpsolve('get_break_at_value', lp_handle) */

static void impl_get_break_at_value()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_break_at_value(lp0);
}


/* name = xxlpsolve('get_col_name', lp_handle, column) */
/* [names] = xxlpsolve('get_col_name', lp_handle) */

static void impl_get_col_name()
{
        char *name;

        if (nrhs == 1+1) {
                int n, i;

                n = get_Ncolumns(lp0);
                {
#if defined HasCellMatrix
                        MatrixEl pa;

                        PrepareCall(cmd, 1, nrhs, NULL);
                        plhs[0] = mxCreateCellMatrix(1, n);
                        for (i = 0; i < n; i++) {
                                name = get_col_name(lp0, i + 1);
                                pa = CreateString(&name, 1, NULL);
                        	mxSetCell(plhs[0], i, pa);
                	}
#else
                	char **names;

                        names = (char **) matCalloc(n, sizeof(*names));
                        for (i = 0; i < n; i++)
                                names[i] = strdup(get_col_name(lp0, i + 1));
                        CreateString(names, n, plhs);
                        for (i = 0; i < n; i++)
                                free(names[i]);
                        matFree(names);
#endif
                }
        }
        else {
        	PrepareCall(cmd, 2, nrhs, NULL);
                name = get_col_name(lp0, (int) GetRealScalar(prhs + 2));
                CreateString(&name, 1, plhs);
        }
}


/* [column, return] = xxlpsolve('get_column', lp_handle, col_nr) */

static void impl_get_column()
{
        int col;

        PrepareCall(cmd, 2, nrhs, NULL);
	col = (int) GetRealScalar(prhs + 2);
        pr = CreateDoubleMatrix(1 + get_Nrows(lp0), 1, plhs);
	result = get_column(lp0, col, pr);
        if (nlhs > 1) {
                pr = CreateDoubleMatrix(1, 1, plhs + 1);
                *pr = (double) result;
        }
}


/* return = xxlpsolve('get_constr_type', lp_handle, row) */
/* [constr_type] = xxlpsolve('get_constr_type', lp_handle) */

static void impl_get_constr_type()
{
	if (nrhs == 1+1) {
                int i, m;

                PrepareCall(cmd, 1, nrhs, NULL);
                m = get_Nrows(lp0);
                pr = CreateDoubleMatrix(m, 1, plhs);
                for (i = 1; i <= m; i++)
                        *(pr++) = get_constr_type(lp0, i);
        }
        else {
        	pr = PrepareCall(cmd, 2, nrhs, plhs);
        	*pr = (double) get_constr_type(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* [constr, return] = xxlpsolve('get_constraints', lp_handle) */

static void impl_get_constraints()
{
        PrepareCall(cmd, 1, nrhs, NULL);
        pr = CreateDoubleMatrix(get_Nrows(lp0), 1, plhs);
	result = get_constraints(lp0, pr);
        if (nlhs > 1) {
                pr = CreateDoubleMatrix(1, 1, plhs + 1);
                *pr = (double) result;
        }
}


/* [duals, return] = xxlpsolve('get_dual_solution', lp_handle) */

static void impl_get_dual_solution()
{
        int i;
	REAL	*vec;

        PrepareCall(cmd, 1, nrhs, NULL);
        i = get_Nrows(lp0) + get_Ncolumns(lp0);
        pr = CreateDoubleMatrix(i, 1, plhs);
	result = get_ptr_dual_solution(lp0, &vec);
        memcpy(pr, vec + 1, i * sizeof(*vec));
        if (nlhs > 1) {
                pr = CreateDoubleMatrix(1, 1, plhs + 1);
                *pr = (double) result;
        }
}


/* return = xxlpsolve('get_epsb', lp_handle) */

static void impl_get_epsb()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_epsb(lp0);
}


/* return = xxlpsolve('get_epsd', lp_handle) */

static void impl_get_epsd()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_epsd(lp0);
}


/* return = xxlpsolve('get_epsel', lp_handle) */

static void impl_get_epsel()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_epsel(lp0);
}


/* return = xxlpsolve('get_epsint', lp_handle) */

static void impl_get_epsint()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_epsint(lp0);
}


/* return = xxlpsolve('get_epsperturb', lp_handle) */

static void impl_get_epsperturb()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_epsperturb(lp0);
}


/* return = xxlpsolve('get_epspivot', lp_handle) */

static void impl_get_epspivot()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_epspivot(lp0);
}


/* return = xxlpsolve('get_improve', lp_handle) */

static void impl_get_improve()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_improve(lp0);
}


/* return = xxlpsolve('get_infinite', lp_handle) */

static void impl_get_infinite()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_infinite(lp0);
}


/* return = xxlpsolve('get_lowbo', lp_handle, column) */
/* [return] = xxlpsolve('get_lowbo', lp_handle) */

static void impl_get_lowbo()
{
	if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
                  *(pr++) = get_lowbo(lp0, i);
        }
        else {
        	pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = get_lowbo(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('get_lp_index', lp_handle, orig_index) */

static void impl_get_lp_index()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
        *pr = (double) get_lp_index(lp0, (int) GetRealScalar(prhs + 2));
}


/* name = xxlpsolve('get_lp_name', lp_handle) */

static void impl_get_lp_name()
{
        char *name;

        PrepareCall(cmd, 1, nrhs, NULL);
        name = get_lp_name(lp0);
        CreateString(&name, 1, plhs);
}


/* value = xxlpsolve('get_mat', lp_handle, row, col) */
/* [matrix, return] = xxlpsolve('get_mat', lp_handle) */

static void impl_get_mat()
{
	if (nrhs == 1+1) {
                int m, n, i;
        	REAL	*vec;

                PrepareCall(cmd, 1, nrhs, NULL);
                m = get_Nrows(lp0);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(m, n, plhs);
                vec = (REAL *) matCalloc(1 + m, sizeof(*vec));
                result = TRUE;
                for (i = 1; (i <= n) && (result); i++) {
        		result = get_column(lp0, i, vec);
                	memcpy(pr, vec + 1, m * sizeof(*vec));
                        pr += m;
                }
                matFree(vec);
                if (nlhs > 1) {
                        pr = CreateDoubleMatrix(1, 1, plhs + 1);
                        *pr = (double) result;
                }
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
        	*pr = get_mat(lp0, (int) GetRealScalar(prhs + 2), (int) GetRealScalar(prhs + 3));
        }
}


/* return = xxlpsolve('get_max_level', lp_handle) */

static void impl_get_max_level()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_max_level(lp0);
}


/* return = xxlpsolve('get_maxpivot', lp_handle) */

static void impl_get_maxpivot()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_maxpivot(lp0);
}


/* return = xxlpsolve('get_mip_gap', lp_handle, absolute) */

static void impl_get_mip_gap()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
        *pr = get_mip_gap(lp0, (MYBOOL) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('get_nameindex', lp_handle, name, isrow) */

static void impl_get_nameindex()
{
        pr = PrepareCall(cmd, 3, nrhs, plhs);
        GetString(prhs + 2, buf, bufsz);
        result = get_nameindex(lp0, buf, (MYBOOL) GetRealScalar(prhs + 3));
	*pr = (double) result;
}


/* return = xxlpsolve('get_Ncolumns', lp_handle) */

static void impl_get_Ncolumns()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_Ncolumns(lp0);
}


/* return = xxlpsolve('get_negrange', lp_handle) */

static void impl_get_negrange()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_negrange(lp0);
}


/* return = xxlpsolve('get_nonzeros', lp_handle) */

static void impl_get_nonzeros()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_nonzeros(lp0);
}


/* return = xxlpsolve('get_Norig_columns', lp_handle) */

static void impl_get_Norig_columns()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_Norig_columns(lp0);
}


/* return = xxlpsolve('get_Norig_rows', lp_handle) */

static void impl_get_Norig_rows()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_Norig_rows(lp0);
}


/* return = xxlpsolve('get_Nrows', lp_handle) */

static void impl_get_Nrows()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_Nrows(lp0);
}


/* return = xxlpsolve('get_obj_bound', lp_handle) */

static void impl_get_obj_bound()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_obj_bound(lp0);
}


/* [row_vec, return] = xxlpsolve('get_obj_fn', lp_handle) */
/* [row_vec, return] = xxlpsolve('get_obj_fun', lp_handle) */

static void impl_get_obj_fn()
{
        int n;
	REAL	*vec;

        PrepareCall(cmd, 1, nrhs, NULL);
        n = get_Ncolumns(lp0);
        pr = CreateDoubleMatrix(1, n, plhs);
        vec = (REAL *) matCalloc(1 + n, sizeof(*vec));
	result = get_row(lp0, 0, vec);
        memcpy(pr, vec + 1, n * sizeof(*vec));
        matFree(vec);
        if (nlhs > 1) {
                pr = CreateDoubleMatrix(1, 1, plhs + 1);
                *pr = (double) result;
        }
}


/* return = xxlpsolve('get_objective', lp_handle) */

static void impl_get_objective()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_objective(lp0);
}


/* name = xxlpsolve('get_objective_name', lp_handle) */

static void impl_get_objective_name()
{
        char *name;

        PrepareCall(cmd, 1, nrhs, NULL);
        name = get_row_name(lp0, 0);
        CreateString(&name, 1, plhs);
}


/* return = xxlpsolve('get_orig_index', lp_handle, lp_index) */

static void impl_get_orig_index()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
        *pr = (double) get_orig_index(lp0, (int) GetRealScalar(prhs + 2));
}


/* name = xxlpsolve('get_origcol_name', lp_handle, column) */
/* [names] = xxlpsolve('get_origcol_name', lp_handle) */

static void impl_get_origcol_name()
{
        char *name;

	if (nrhs == 1+1) {
                int n, i;

                n = get_Ncolumns(lp0);
                {
#if defined HasCellMatrix
                        MatrixEl pa;

                        PrepareCall(cmd, 1, nrhs, NULL);
                        plhs[0] = mxCreateCellMatrix(1, n);
                        for (i = 0; i < n; i++) {
                                name = get_origcol_name(lp0, i + 1);
                                pa = CreateString(&name, 1, NULL);
                        	mxSetCell(plhs[0], i, pa);
                	}
#else
                	char **names;

                        names = (char **) matCalloc(n, sizeof(*names));
                        for (i = 0; i < n; i++)
                                names[i] = strdup(get_origcol_name(lp0, i + 1));
                        CreateString(names, n, plhs);
                        for (i = 0; i < n; i++)
                                free(names[i]);
                        matFree(names);
#endif
                }
        }
        else {
	        PrepareCall(cmd, 2, nrhs, NULL);
                name = get_origcol_name(lp0, (int) GetRealScalar(prhs + 2));
                CreateString(&name, 1, plhs);
        }
}


/* name = xxlpsolve('get_origrow_name', lp_handle, row) */
/* [names] = xxlpsolve('get_origrow_name', lp_handle) */

static void impl_get_origrow_name()
{
        char *name;

	if (nrhs == 1+1) {
                int m, i;

                m = get_Nrows(lp0);
                {
#if defined HasCellMatrix
                        MatrixEl pa;

                        PrepareCall(cmd, 1, nrhs, NULL);
                        plhs[0] = mxCreateCellMatrix(1, m);
                        for (i = 0; i < m; i++) {
                                name = get_origrow_name(lp0, i + 1);
                                pa = CreateString(&name, 1, NULL);
                        	mxSetCell(plhs[0], i, pa);
                	}
#else
                	char **names;

                        names = (char **) matCalloc(m, sizeof(*names));
                        for (i = 0; i < m; i++)
                                names[i] = strdup(get_origrow_name(lp0, i + 1));
                        CreateString(names, m, plhs);
                        for (i = 0; i < m; i++)
                                free(names[i]);
                        matFree(names);
#endif
                }
        }
        else {
        	PrepareCall(cmd, 2, nrhs, NULL);
                name = get_origrow_name(lp0, (int) GetRealScalar(prhs + 2));
                CreateString(&name, 1, plhs);
        }
}


/* return = xxlpsolve('get_pivoting', lp_handle) */

static void impl_get_pivoting()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_pivoting(lp0);
}


/* return = xxlpsolve('get_presolve', lp_handle) */

static void impl_get_presolve()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_presolve(lp0);
}


/* [pv, return] = xxlpsolve('get_primal_solution', lp_handle) */

static void impl_get_primal_solution()
{
        int i;

        PrepareCall(cmd, 1, nrhs, NULL);
        i = 1 + get_Nrows(lp0) + get_Ncolumns(lp0);
        pr = CreateDoubleMatrix(i, 1, plhs);
	result = get_primal_solution(lp0, pr);
        if (nlhs > 1) {
                pr = CreateDoubleMatrix(1, 1, plhs + 1);
                *pr = (double) result;
        }
}


/* return = xxlpsolve('get_print_sol', lp_handle) */

static void impl_get_print_sol()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_print_sol(lp0);
}


/* return = xxlpsolve('get_rh', lp_handle, row) */
/* [rh] = xxlpsolve('get_rh', lp_handle) */

static void impl_get_rh()
{
	if (nrhs == 1+1) {
                int m, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                m = get_Nrows(lp0);
                pr = CreateDoubleMatrix(1 + m, 1, plhs);
                for (i = 0; i <= m; i++)
                  *(pr++) = get_rh(lp0, i);
        }
        else {
                pr = PrepareCall(cmd, 2, nrhs, plhs);
        	*pr = get_rh(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('get_rh_range', lp_handle, row) */
/* [rh_ranges] = xxlpsolve('get_rh_range', lp_handle) */

static void impl_get_rh_range()
{
	if (nrhs == 1+1) {
                int m, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                m = get_Nrows(lp0);
                pr = CreateDoubleMatrix(m, 1, plhs);
                for (i = 1; i <= m; i++)
                  *(pr++) = get_rh_range(lp0, i);
        }
        else {
	        pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = get_rh_range(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* [row, return] = xxlpsolve('get_row', lp_handle, row_nr) */

static void impl_get_row()
{
        int n;
	REAL	*vec;

        PrepareCall(cmd, 2, nrhs, NULL);
        n = get_Ncolumns(lp0);
        pr = CreateDoubleMatrix(1, n, plhs);
        vec = (REAL *) matCalloc(1 + n, sizeof(*vec));
	result = get_row(lp0, (int) GetRealScalar(prhs + 2), vec);
        memcpy(pr, vec + 1, n * sizeof(*vec));
        matFree(vec);
        if (nlhs > 1) {
                pr = CreateDoubleMatrix(1, 1, plhs + 1);
                *pr = (double) result;
        }
}


/* name = xxlpsolve('get_row_name', lp_handle, row) */
/* [names] = xxlpsolve('get_row_name', lp_handle) */

static void impl_get_row_name()
{
        char *name;

        if (nrhs == 1+1) {
                int m, i;

                m = get_Nrows(lp0);
                {
#if defined HasCellMatrix
                        MatrixEl pa;

                        PrepareCall(cmd, 1, nrhs, NULL);
                        plhs[0] = mxCreateCellMatrix(1, m);
                        for (i = 0; i < m; i++) {
                                name = get_row_name(lp0, i + 1);
                                pa = CreateString(&name, 1, NULL);
                        	mxSetCell(plhs[0], i, pa);
                	}
#else
                	char **names;

                        names = (char **) matCalloc(m, sizeof(*names));
                        for (i = 0; i < m; i++)
                                names[i] = strdup(get_row_name(lp0, i + 1));
                        CreateString(names, m, plhs);
                        for (i = 0; i < m; i++)
                                free(names[i]);
                        matFree(names);
#endif
                }
        }
        else {
                PrepareCall(cmd, 2, nrhs, NULL);
                name = get_row_name(lp0, (int) GetRealScalar(prhs + 2));
                CreateString(&name, 1, plhs);
        }
}


/* return = xxlpsolve('get_scalelimit', lp_handle) */

static void impl_get_scalelimit()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_scalelimit(lp0);
}


/* return = xxlpsolve('get_scaling', lp_handle) */

static void impl_get_scaling()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_scaling(lp0);
}


/* [objfrom, objtill, objfromvalue, objtillvalue, return] = xxlpsolve('get_sensitivity_obj', lp_handle) */
/* [objfrom, objtill, objfromvalue, objtillvalue, return] = xxlpsolve('get_sensitivity_objex', lp_handle) */

static void impl_get_sensitivity_objex()
{
        int n;
	REAL	*objfrom, *objtill, *objfromvalue, *objtillvalue;

        PrepareCall(cmd, 1, nrhs, NULL);
        n = get_Ncolumns(lp0);
        objfrom = CreateDoubleMatrix(1, n, plhs);
        if (nlhs > 1)
        	objtill = CreateDoubleMatrix(1, n, plhs + 1);
        else
                objtill = NULL;
        if (nlhs > 2)
        	objfromvalue = CreateDoubleMatrix(1, n, plhs + 2);
        else
                objfromvalue = NULL;
        if (nlhs > 3) {
        	objtillvalue = CreateDoubleMatrix(1, n, plhs + 3);
                memset(objtillvalue, 0, n * sizeof(*objtillvalue));
        }
        else
                objtillvalue = NULL;
	result = get_sensitivity_objex(lp0, objfrom, objtill, objfromvalue, NULL /* objtillvalue */);
        if (nlhs > 4) {
                pr = CreateDoubleMatrix(1, 1, plhs + 4);
                *pr = (double) result;
        }
}


/* [duals, dualsfrom, dualstill, return] = xxlpsolve('get_sensitivity_rhs', lp_handle) */
/* [duals, dualsfrom, dualstill, return] = xxlpsolve('get_sensitivity_rhsex', lp_handle) */

static void impl_get_sensitivity_rhsex()
{
        int i;
        REAL *duals, *dualsfrom, *dualstill;

        PrepareCall(cmd, 1, nrhs, NULL);
        i = get_Nrows(lp0) + get_Ncolumns(lp0);
        duals = CreateDoubleMatrix(i, 1, plhs);
        if (nlhs > 1)
        	dualsfrom = CreateDoubleMatrix(i, 1, plhs + 1);
        else
		dualsfrom = NULL;
        if (nlhs > 2)
        	dualstill = CreateDoubleMatrix(i, 1, plhs + 2);
        else
                dualstill = NULL;
	result = get_sensitivity_rhs(lp0, duals, dualsfrom, dualstill);
        if (nlhs > 3) {
                pr = CreateDoubleMatrix(1, 1, plhs + 3);
                *pr = (double) result;
        }
}


/* return = xxlpsolve('get_simplextype', lp_handle) */

static void impl_get_simplextype()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_simplextype(lp0);
}


/* [obj, x, duals, return] = xxlpsolve('get_solution', lp_handle) */

static void impl_get_solution()
{
        int m, n;
        REAL *duals;

        PrepareCall(cmd, 1, nrhs, NULL);

        pr = CreateDoubleMatrix(1, 1, plhs);
        *pr = get_objective(lp0);

        if (nlhs > 1) {
	        n = get_Ncolumns(lp0);
        	pr = CreateDoubleMatrix(n, 1, plhs + 1);
	        result = get_variables(lp0, pr);
        }

        if (nlhs > 2) {
                m = get_Nrows(lp0);
        	pr = CreateDoubleMatrix(m, 1, plhs + 2);
		result &= get_ptr_dual_solution(lp0, &duals);
                memcpy(pr, duals + 1, m * sizeof(*pr));
        }

        if (nlhs > 3) {
                pr = CreateDoubleMatrix(1, 1, plhs + 3);
                *pr = (double) result;
        }
}


/* return = xxlpsolve('get_solutioncount', lp_handle) */

static void impl_get_solutioncount()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_solutioncount(lp0);
}


/* return = xxlpsolve('get_solutionlimit', lp_handle) */

static void impl_get_solutionlimit()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_solutionlimit(lp0);
}


/* return = xxlpsolve('get_status', lp_handle) */

static void impl_get_status()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_status(lp0);
}


/* return = xxlpsolve('get_statustext', lp_handle, statuscode) */

static void impl_get_statustext()
{
        char *name;

        PrepareCall(cmd, 2, nrhs, NULL);
        name = get_statustext(lp0, (int) GetRealScalar(prhs + 2));
	CreateString(&name, 1, plhs);
}


/* return = xxlpsolve('get_timeout', lp_handle) */

static void impl_get_timeout()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_timeout(lp0);
}


/* return = xxlpsolve('get_total_iter', lp_handle) */

static void impl_get_total_iter()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_total_iter(lp0);
}


/* return = xxlpsolve('get_total_nodes', lp_handle) */

static void impl_get_total_nodes()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) get_total_nodes(lp0);
}


/* return = xxlpsolve('get_upbo', lp_handle, column) */
/* [upbo] = xxlpsolve('get_upbo', lp_handle) */

static void impl_get_upbo()
{
	if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
                  *(pr++) = get_upbo(lp0, i);
        }
        else {
	        pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = get_upbo(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('get_var_branch', lp_handle, column) */
/* [var_branch] = xxlpsolve('get_var_branch', lp_handle) */

static void impl_get_var_branch()
{
	if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
                  *(pr++) = get_var_branch(lp0, i);
        }
        else {
                pr = PrepareCall(cmd, 2, nrhs, plhs);
        	*pr = (double) get_var_branch(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('get_var_dualresult', lp_handle, index) */

static void impl_get_var_dualresult()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = get_var_dualresult(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('get_var_primalresult', lp_handle, index) */

static void impl_get_var_primalresult()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = get_var_primalresult(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('get_var_priority', lp_handle, column) */
/* [var_priority] = xxlpsolve('get_var_priority', lp_handle) */

static void impl_get_var_priority()
{
	if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
                  *(pr++) = get_var_priority(lp0, i);
        }
        else {
        	pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = (double) get_var_priority(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* [var, return] = xxlpsolve('get_variables', lp_handle) */

static void impl_get_variables()
{
        int n;

        PrepareCall(cmd, 1, nrhs, NULL);
        n = get_Ncolumns(lp0);
        pr = CreateDoubleMatrix(n, 1, plhs);
        result = get_variables(lp0, pr);
        if (nlhs > 1) {
                pr = CreateDoubleMatrix(1, 1, plhs + 1);
                *pr = (double) result;
        }
}


/* return = xxlpsolve('get_verbose', lp_handle) */

static void impl_get_verbose()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
	*pr = (double) get_verbose(lp0);
}


/* return = xxlpsolve('get_working_objective', lp_handle) */

static void impl_get_working_objective()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = get_working_objective(lp0);
}


/* return = xxlpsolve('has_BFP', lp_handle) */

static void impl_has_BFP()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) has_BFP(lp0);
}


/* return = xxlpsolve('has_XLI', lp_handle) */

static void impl_has_XLI()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) has_XLI(lp0);
}


/* return = xxlpsolve('is_add_rowmode', lp_handle) */

static void impl_is_add_rowmode()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) is_add_rowmode(lp0);
}


/* return = xxlpsolve('is_anti_degen', lp_handle, testmask) */

static void impl_is_anti_degen()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) is_anti_degen(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('is_binary', lp_handle, column) */
/* [binary] = xxlpsolve('is_binary', lp_handle) */

static void impl_is_binary()
{
        if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
        		*(pr++) = is_binary(lp0, i);
        }
        else {
        	pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = (double) is_binary(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('is_break_at_first', lp_handle) */

static void impl_is_break_at_first()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) is_break_at_first(lp0);
}


/* return = xxlpsolve('is_constr_type', lp_handle, row, mask) */

static void impl_is_constr_type()
{
        pr = PrepareCall(cmd, 3, nrhs, plhs);
	*pr = (double) is_constr_type(lp0, (int) GetRealScalar(prhs + 2), (int) GetRealScalar(prhs + 3));
}


/* return = xxlpsolve('is_debug', lp_handle) */

static void impl_is_debug()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) is_debug(lp0);
}


/* return = xxlpsolve('is_feasible', lp_handle, [values] {, threshold}) */

static void impl_is_feasible()
{
        int i, n;
	REAL	*vec, threshold;

        if (nrhs == 2+1)
                n = 2;
        else
                n = 3;
        pr = PrepareCall(cmd, n, nrhs, plhs);
        i = get_Nrows(lp0) + get_Ncolumns(lp0);
        vec = (REAL *) matCalloc(1 + i, sizeof(REAL));
	GetRealVector(prhs + 2, vec, 1, i, TRUE);
        if (n == 2)
                threshold = get_epsint(lp0);
        else
                threshold = GetRealScalar(prhs + 3);
        result = is_feasible(lp0, vec, threshold);
        *pr = (double) result;
	matFree(vec);
}


/* return = xxlpsolve('is_free', lp_handle, column) */
/* [free] = xxlpsolve('is_free', lp_handle) */

static void impl_is_free()
{
        if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
        		*(pr++) = is_free(lp0, i);
        }
        else {
        	pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = (double) is_free(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('is_infinite', lp_handle, value) */

static void impl_is_infinite()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) is_infinite(lp0, GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('is_int', lp_handle, column) */
/* [int] = xxlpsolve('is_int', lp_handle) */

static void impl_is_int()
{
        if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
        		*(pr++) = is_int(lp0, i);
        }
        else {
                pr = PrepareCall(cmd, 2, nrhs, plhs);
        	*pr = (double) is_int(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('is_integerscaling', lp_handle) */

static void impl_is_integerscaling()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) is_integerscaling(lp0);
}


/* return = xxlpsolve('is_maxim', lp_handle) */

static void impl_is_maxim()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) is_maxim(lp0);
}


/* return = xxlpsolve('is_nativeBFP', lp_handle) */

static void impl_is_nativeBFP()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) is_nativeBFP(lp0);
}


/* return = xxlpsolve('is_nativeXLI', lp_handle) */

static void impl_is_nativeXLI()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) is_nativeXLI(lp0);
}


/* return = xxlpsolve('is_negative', lp_handle, column) */
/* [negative] = xxlpsolve('is_negative', lp_handle) */

static void impl_is_negative()
{
        if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
        		*(pr++) = is_negative(lp0, i);
        }
        else {
	        pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = (double) is_negative(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('is_piv_mode', lp_handle, testmask) */

static void impl_is_piv_mode()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) is_piv_mode(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('is_piv_rule', lp_handle, rule) */

static void impl_is_piv_rule()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) is_piv_rule(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('is_presolve', lp_handle, testmask) */

static void impl_is_presolve()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) is_presolve(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('is_scalemode', lp_handle, testmask) */

static void impl_is_scalemode()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) is_scalemode(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('is_scaletype', lp_handle, scaletype) */

static void impl_is_scaletype()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) is_scaletype(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('is_semicont', lp_handle, column) */
/* [semicont] = xxlpsolve('is_semicont', lp_handle) */

static void impl_is_semicont()
{
        if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
        		*(pr++) = is_semicont(lp0, i);
        }
        else {
        	pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = (double) is_semicont(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('is_SOS_var', lp_handle, column) */
/* [SOS_var] = xxlpsolve('is_SOS_var', lp_handle) */

static void impl_is_SOS_var()
{
        if (nrhs == 1+1) {
                int n, i;

                PrepareCall(cmd, 1, nrhs, NULL);
                n = get_Ncolumns(lp0);
                pr = CreateDoubleMatrix(n, 1, plhs);
                for (i = 1; i <= n; i++)
        		*(pr++) = is_SOS_var(lp0, i);
        }
        else {
        	pr = PrepareCall(cmd, 2, nrhs, plhs);
		*pr = (double) is_SOS_var(lp0, (int) GetRealScalar(prhs + 2));
        }
}


/* return = xxlpsolve('is_trace', lp_handle) */

static void impl_is_trace()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = (double) is_nativeXLI(lp0);
}


/* versionstring = xxlpsolve('lp_solve_version') */

static void impl_lp_solve_version()
{
        int majorversion, minorversion, release, build;

        PrepareCall(cmd, 0, nrhs, NULL);
        lp_solve_version(&majorversion, &minorversion, &release, &build);
        sprintf(buf, "%d.%d.%d.%d", majorversion, minorversion, release, build);
	CreateString(&buf, 1, plhs);
}


/* lp_handle = xxlpsolve('make_lp', rows, columns) */

static void impl_make_lp()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) create_handle(make_lp((int) GetRealScalar(prhs + 1), (int) GetRealScalar(prhs + 2)), "make_lp failed");
}


/* xxlpsolve('print_constraints', lp_handle {, columns}) */

static void impl_print_constraints()
{
        int n, columns;

        if (nrhs == 1+1)
        	n = 1;
        else
        	n = 2;
        PrepareCall(cmd, n, nrhs, NULL);
        if (n == 1)
                columns = 1;
        else
                columns = (int) GetRealScalar(prhs + 2);
	print_constraints(lp0, columns);
}


/* return = xxlpsolve('print_debugdump', lp_handle, filename) */

static void impl_print_debugdump()
{
        char filename[260];

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        GetString(prhs + 2, filename, sizeof(filename));
        result = print_debugdump(lp0, filename);
	*pr = (double) result;
}


/* xxlpsolve('print_duals', lp_handle) */

static void impl_print_duals()
{
        PrepareCall(cmd, 1, nrhs, NULL);
	print_duals(lp0);
}


/* xxlpsolve('print_lp', lp_handle) */

static void impl_print_lp()
{
        PrepareCall(cmd, 1, nrhs, NULL);
	print_lp(lp0);
}


/* xxlpsolve('print_objective', lp_handle) */

static void impl_print_objective()
{
        PrepareCall(cmd, 1, nrhs, NULL);
	print_objective(lp0);
}


/* xxlpsolve('print_scales', lp_handle) */

static void impl_print_scales()
{
        PrepareCall(cmd, 1, nrhs, NULL);
	print_scales(lp0);
}


/* xxlpsolve('print_solution', lp_handle {, columns}) */

static void impl_print_solution()
{
        int n, columns;

        if (nrhs == 1+1)
        	n = 1;
        else
        	n = 2;
        PrepareCall(cmd, n, nrhs, NULL);
        if (n == 1)
                columns = 1;
        else
                columns = (int) GetRealScalar(prhs + 2);
	print_solution(lp0, columns);
}


/* xxlpsolve('print_str', lp_handle, str) */

static void impl_print_str()
{
        PrepareCall(cmd, 2, nrhs, NULL);
        GetString(prhs + 2, buf, bufsz);
        print_str(lp0, buf);
}


/* xxlpsolve('print_tableau', lp_handle) */

static void impl_print_tableau()
{
        PrepareCall(cmd, 1, nrhs, NULL);
	print_tableau(lp0);
}


/* [handle_vec] = xxlpsolve('print_handle') */
/* print all used handles */

static void impl_print_handle()
{
        int i, j, k;

	j = 0;
        for (i = 0; i <= lp_last; i++)
	  if (lp[i] != NULL)
	     j++;
        if (j)
          k = 1;
        else
          k = 0;
        pr = CreateDoubleMatrix(j, k, plhs);
	k = 0;
	for (i = 0; i <= lp_last; i++)
	  if (lp[i] != NULL)
            pr[k++] = i;
}


/* lp_handle = xxlpsolve('read_freeMPS', filename {, verbose}) */

static void impl_read_freeMPS()
{
        int n, verbose;
        char filename[260];

        if (nrhs == 1+1)
        	n = 1;
        else
        	n = 2;
        pr = PrepareCall(cmd, n, nrhs, plhs);
        if (n >= 2)
        	verbose = (int) GetRealScalar(prhs + 2);
        else
                verbose = NORMAL;
	GetString(prhs + 1, filename, sizeof(filename));
        *pr = (double) create_handle(read_freeMPS(filename, verbose), "read_freeMPS can't read file.");
}


/* lp_handle = xxlpsolve('read_lp_file', filename {, verbose {, lp_name}}) */
/* lp_handle = xxlpsolve('read_lp', filename {, verbose {, lp_name}}) */
/* lp_handle = xxlpsolve('read_LP', filename {, verbose {, lp_name}}) */

static void impl_read_LP()
{
        int n, verbose;
        char filename[260], lp_name[50];

        if (nrhs == 1+1)
        	n = 1;
        else if (nrhs == 1+2)
        	n = 2;
        else
        	n = 3;
        pr = PrepareCall(cmd, n, nrhs, plhs);
	GetString(prhs + 1, filename, sizeof(filename));
        if (n >= 2)
        	verbose = (int) GetRealScalar(prhs + 2);
        else
                verbose = NORMAL;
        if (n >= 3)
        	GetString(prhs + 3, lp_name, sizeof(lp_name));
        else
                *lp_name = 0;
        lp0 = read_LP(filename, verbose, lp_name);
        *pr = (double) create_handle(lp0, "read_LP can't read file.");
}


/* lp_handle = xxlpsolve('read_mps', filename {, verbose}) */
/* lp_handle = xxlpsolve('read_MPS', filename {, verbose}) */

static void impl_read_MPS()
{
        int n, verbose;
        char filename[260];

        if (nrhs == 1+1)
        	n = 1;
        else
        	n = 2;
        pr = PrepareCall(cmd, n, nrhs, plhs);
	GetString(prhs + 1, filename, sizeof(filename));
        if (n >= 2)
        	verbose = (int) GetRealScalar(prhs + 2);
        else
                verbose = NORMAL;
        *pr = (double) create_handle(read_MPS(filename, verbose), "read_MPS can't read file.");
}


/* lp_handle = xxlpsolve('read_XLI', xliname, modelname {, dataname {, options {, verbose}}} */

static void impl_read_XLI()
{
        int n, verbose;
        char xliname[260], modelname[260], dataname[260], options[260];

        if (nrhs == 1+2)
        	n = 2;
        else if (nrhs == 1+3)
        	n = 3;
        else if (nrhs == 1+4)
        	n = 4;
        else
        	n = 5;
        pr = PrepareCall(cmd, n, nrhs, plhs);
	GetString(prhs + 1, xliname, sizeof(xliname));
        GetString(prhs + 2, modelname, sizeof(modelname));
        if (n >= 3)
        	GetString(prhs + 3, dataname, sizeof(dataname));
        else
                *dataname = 0;
        if (n >= 4)
        	GetString(prhs + 4, options, sizeof(options));
        else
                *options = 0;
        if (n >= 5)
        	verbose = (int) GetRealScalar(prhs + 5);
        else
                verbose = NORMAL;
        *pr = (double) create_handle(read_XLI(xliname, modelname, (*dataname) ? dataname : NULL, options, verbose), "read_XLI can't read file.");
}


/* return = xxlpsolve('set_add_rowmode', lp_handle, turnon) */

static void impl_set_add_rowmode()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	*pr = (double) set_add_rowmode(lp0, (MYBOOL) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_anti_degen', lp_handle, anti_degen) */

static void impl_set_anti_degen()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_anti_degen(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_basis', lp_handle, [bascolumn], nonbasic) */

static void impl_set_basis()
{
        int i, *bascolumn;
        MYBOOL nonbasic;

        pr = PrepareCall(cmd, 3, nrhs, plhs);
        nonbasic = (MYBOOL) GetRealScalar(prhs + 3);
        i = get_Nrows(lp0) + ((nonbasic) ? get_Ncolumns(lp0) : 0);
        bascolumn = (int *) matCalloc(1 + i, sizeof(*bascolumn));
	GetIntVector(prhs + 2, bascolumn, 1, i, TRUE);
	result = set_basis(lp0, bascolumn, nonbasic);
        *pr = (double) result;
	matFree(bascolumn);
}


/* xxlpsolve('set_basiscrash', lp_handle, mode) */

static void impl_set_basiscrash()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_basiscrash(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_bb_depthlimit', lp_handle, bb_maxlevel) */

static void impl_set_bb_depthlimit()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_bb_depthlimit(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_bb_floorfirst', lp_handle, bb_floorfirst) */

static void impl_set_bb_floorfirst()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_bb_floorfirst(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_bb_rule', lp_handle, bb_rule) */

static void impl_set_bb_rule()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_bb_rule(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_BFP', lp_handle, filename) */

static void impl_set_BFP()
{
        char filename[260];

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        GetString(prhs + 2, filename, sizeof(filename));
        result = set_BFP(lp0, filename);
	*pr = (double) result;
}


/* return = xxlpsolve('set_binary', lp_handle, column, must_be_bin) */
/* return = xxlpsolve('set_binary', lp_handle, [must_be_bin]) */

static void impl_set_binary()
{
        if (nrhs == 1+2) {
                int i, n, *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                n = get_Ncolumns(lp0);
                vec = (int *) matCalloc(n, sizeof(*vec));
        	GetIntVector(prhs + 2, vec, 0, n, TRUE);
                result = TRUE;
                for (i = 0; (i < n) && (result); i++)
                        result = set_binary(lp0, i + 1, (MYBOOL) vec[i]);
                matFree(vec);
        }
        else {
                pr = PrepareCall(cmd, 3, nrhs, plhs);
        	result = set_binary(lp0, (int) GetRealScalar(prhs + 2), (MYBOOL) GetRealScalar(prhs + 3));
        }
        *pr = (double) result;
}


/* return = xxlpsolve('set_bounds', lp_handle, column, lower, upper) */
/* return = xxlpsolve('set_bounds', lp_handle, [lower], [upper]) */

static void impl_set_bounds()
{
        if (nrhs == 1+3) {
                int i, n;
                REAL	*lower, *upper;

                pr = PrepareCall(cmd, 3, nrhs, plhs);
                n = get_Ncolumns(lp0);
                lower = (REAL *) matCalloc(n, sizeof(REAL));
                upper = (REAL *) matCalloc(n, sizeof(REAL));
        	GetRealVector(prhs + 2, lower, 0, n, TRUE);
                GetRealVector(prhs + 3, upper, 0, n, TRUE);
                result = TRUE;
                for (i = 0; (i < n) && (result); i++)
                        result = set_bounds(lp0, i + 1, lower[i], upper[i]);
                matFree(upper);
                matFree(lower);
        }
        else {
	        pr = PrepareCall(cmd, 4, nrhs, plhs);
		result = set_bounds(lp0, (int) GetRealScalar(prhs + 2), GetRealScalar(prhs + 3), GetRealScalar(prhs + 4));
        }
        *pr = (double) result;
}


/* xxlpsolve('set_bounds_tighter', lp_handle, tighten) */

static void impl_set_bounds_tighter()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_bounds_tighter(lp0, (MYBOOL) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_break_at_first', lp_handle, break_at_first) */

static void impl_set_break_at_first()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_break_at_first(lp0, (MYBOOL) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_break_at_value', lp_handle, break_at_value) */

static void impl_set_break_at_value()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_break_at_value(lp0, GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_col_name', lp_handle, column, name) */
/* return = xxlpsolve('set_col_name', lp_handle, [names]) */

static void impl_set_col_name()
{
        if (nrhs == 1+2) {
                int n, i;
                strArray pa;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                n = get_Ncolumns(lp0);
                pa = GetCellCharItems(prhs + 2, n);
                result = TRUE;
                for (i = 0; (i < n) && (result); i++) {
                	GetCellString(pa + i, buf, bufsz);
                        result = set_col_name(lp0, i + 1, buf);
                }
                FreeCellCharItems(pa, n);
        }
        else {
                pr = PrepareCall(cmd, 3, nrhs, plhs);
                GetString(prhs + 3, buf, bufsz);
                result = set_col_name(lp0, (int) GetRealScalar(prhs + 2), buf);
        }
	*pr = (double) result;
}


/* since always a matrix vector is provided to both set_column and set_columnex, always call the more
   performant sparse version of the two routines */
/* return = xxlpsolve('set_column', lp_handle, col_no, [column]) */
/* return = xxlpsolve('set_columnex', lp_handle, col_no, [column]) */

static void impl_set_column()
{
        int m, count;
        int	*index;
	REAL	*vec;

        pr = PrepareCall(cmd, 3, nrhs, plhs);
        m = get_Nrows(lp0);
        vec = (REAL *) matCalloc(1 + m, sizeof(*vec));
        index = (int *) matCalloc(1 + m, sizeof(*index));
	count = GetRealSparseVector(prhs + 3, vec, index, 0, 1 + m, 0);
	result = set_columnex(lp0, (int) GetRealScalar(prhs + 2), count, vec, index);
        *pr = (double) result;
        matFree(index);
        matFree(vec);
}


/* return = xxlpsolve('set_constr_type', lp_handle, row, con_type) */
/* return = xxlpsolve('set_constr_type', lp_handle, [con_type]) */

static void impl_set_constr_type()
{
        if (nrhs == 1+2) {
                int i, m, *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                m = get_Nrows(lp0);
                vec = (int *) matCalloc(m, sizeof(*vec));
        	GetIntVector(prhs + 2, vec, 0, m, TRUE);
                result = TRUE;
                for (i = 0; (i < m) && (result); i++)
                        result = set_constr_type(lp0, i + 1, vec[i]);
                matFree(vec);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_constr_type(lp0, (int) GetRealScalar(prhs + 2), (int) GetRealScalar(prhs + 3));
        }
        *pr = (double) result;
}


/* xxlpsolve('set_debug', lp_handle, debug) */

static void impl_set_debug()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_debug(lp0, (MYBOOL) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_epsb', lp_handle, epsb) */

static void impl_set_epsb()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_epsb(lp0, GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_epsd', lp_handle, epsd) */

static void impl_set_epsd()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_epsd(lp0, GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_epsel', lp_handle, epsel) */

static void impl_set_epsel()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_epsel(lp0, GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_epsint', lp_handle, epsint) */

static void impl_set_epsint()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_epsint(lp0, GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_epsperturb', lp_handle, epsperturb) */

static void impl_set_epsperturb()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_epsperturb(lp0, GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_epspivot', lp_handle, epspivot) */

static void impl_set_epspivot()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_epspivot(lp0, GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_free', lp_handle, column) */

static void impl_set_free()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
	result = set_free(lp0, (int) GetRealScalar(prhs + 2));
        *pr = (double) result;
}


/* xxlpsolve('set_improve', lp_handle, improve) */

static void impl_set_improve()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_improve(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_infinite', lp_handle, infinite) */

static void impl_set_infinite()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_infinite(lp0, GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_int', lp_handle, column, must_be_int) */
/* return = xxlpsolve('set_int', lp_handle, [must_be_int]) */

static void impl_set_int()
{
        if (nrhs == 1+2) {
                int i, n, *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                n = get_Ncolumns(lp0);
                vec = (int *) matCalloc(n, sizeof(*vec));
        	GetIntVector(prhs + 2, vec, 0, n, TRUE);
                result = TRUE;
                for (i = 0; (i < n) && (result); i++)
                        result = set_int(lp0, i + 1, (MYBOOL) vec[i]);
                matFree(vec);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_int(lp0, (int) GetRealScalar(prhs + 2), (MYBOOL) GetRealScalar(prhs + 3));
        }
        *pr = (double) result;
}


/* return = xxlpsolve('set_lowbo', lp_handle, column, value) */
/* return = xxlpsolve('set_lowbo', lp_handle, [values]) */

static void impl_set_lowbo()
{
        if (nrhs == 1+2) {
                int i, n;
                REAL *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                n = get_Ncolumns(lp0);
                vec = (REAL *) matCalloc(n, sizeof(*vec));
        	GetRealVector(prhs + 2, vec, 0, n, TRUE);
                result = TRUE;
                for (i = 0; (i < n) && (result); i++)
                        result = set_lowbo(lp0, i + 1, vec[i]);
                matFree(vec);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_lowbo(lp0, (int) GetRealScalar(prhs + 2), GetRealScalar(prhs + 3));
        }
        *pr = (double) result;
}


/* return = xxlpsolve('set_lp_name', lp_handle, name) */

static void impl_set_lp_name()
{
        pr = PrepareCall(cmd, 2, nrhs, plhs);
        GetString(prhs + 2, buf, bufsz);
        result = set_lp_name(lp0, buf);
	*pr = (double) result;
}


/* return = xxlpsolve('set_mat', lp_handle, [matrix]) */
/* return = xxlpsolve('set_mat', lp_handle, row, column, value) */

static void impl_set_mat()
{
        if (nrhs == 1+2) {
                int m, n, j, *index, *index1, count;
                REAL *obj, *obj1, *vec, *vec1, a;
                Matrix mat = GetpMatrix(prhs + 2);

		/* Called with a matrix argument */
		m = GetM(mat);
		n = GetN(mat);
		if ((get_Nrows(lp0) != m) || (get_Ncolumns(lp0) != n)
		   || !IsNumeric(mat) || IsComplex(mat)) {
			ErrMsgTxt("invalid argument.");
		}

                obj = obj1 = (REAL *) matCalloc(1 + n, sizeof(*obj));
                result = get_row(lp0, 0, obj);
                vec = (REAL *) matCalloc(1 + m, sizeof(*vec));
                index = (int *) matCalloc(1 + m, sizeof(*index));
                for (j = 1; (j <= n) && (result); j++) {
                        vec1 = vec;
                        index1 = index;
                        count = 0;
                        if ((a = (*(++obj1))) != 0.0) {
                                *(vec1++) = a;
                                *(index1++) = 0;
                                count ++;
                        }
			count += GetRealSparseVector(prhs + 2, vec1, index1, 1, m, j);
                        result = set_columnex(lp0, j, count, vec, index);
                }
                matFree(index);
                matFree(vec);
                matFree(obj);
                pr = PrepareCall(cmd, 2, nrhs, plhs);
        }
        else { /* called with a single matrix element */
        	pr = PrepareCall(cmd, 4, nrhs, plhs);
		result = set_mat(lp0, (int) GetRealScalar(prhs + 2), (int) GetRealScalar(prhs + 3), GetRealScalar(prhs + 4));
        }
        *pr = (double) result;
}


/* xxlpsolve('set_maxim', lp_handle) */

static void impl_set_maxim()
{
        PrepareCall(cmd, 1, nrhs, NULL);
        set_maxim(lp0);
}


/* xxlpsolve('set_maxpivot', max_num_inv) */

static void impl_set_maxpivot()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_maxpivot(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_minim', lp_handle) */

static void impl_set_minim()
{
        PrepareCall(cmd, 1, nrhs, NULL);
        set_minim(lp0);
}


/* xxlpsolve('set_mip_gap', lp_handle, absolute, mip_gap) */

static void impl_set_mip_gap()
{
        PrepareCall(cmd, 3, nrhs, NULL);
	set_mip_gap(lp0, (MYBOOL) GetRealScalar(prhs + 2), GetRealScalar(prhs + 3));
}


/* xxlpsolve('set_negrange', negrange) */

static void impl_set_negrange()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_negrange(lp0, GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_obj', lp_handle, column, value) */
/* return = xxlpsolve('set_obj', lp_handle, [values]) */

static void impl_set_obj()
{
        if (nrhs == 1+2) {
                impl_set_obj_fn();
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_obj(lp0, (int) GetRealScalar(prhs + 2), GetRealScalar(prhs + 3));
	        *pr = (double) result;
        }
}


/* xxlpsolve('set_obj_bound', lp_handle, obj_bound) */

static void impl_set_obj_bound()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_obj_bound(lp0, GetRealScalar(prhs + 2));
}


/* since always a matrix vector is provided to both set_obj_fn and set_obj_fnex, always call the more
   performant sparse version of the two routines */
/* return = xxlpsolve('set_obj_fn', lp_handle, [row]) */
/* return = xxlpsolve('set_obj_fnex', lp_handle, [row]) */

static void impl_set_obj_fn()
{
        int n, count;
        int	*index;
	REAL	*vec;

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        n = get_Ncolumns(lp0);
        vec = (REAL *) matCalloc(1 + n, sizeof(*vec));
        index = (int *) matCalloc(1 + n, sizeof(*index));
	count = GetRealSparseVector(prhs + 2, vec, index, 1, n, 0);
	result = set_obj_fnex(lp0, count, vec, index);
        *pr = (double) result;
        matFree(index);
        matFree(vec);
}


/* return = xxlpsolve('set_outputfile', lp_handle, filename) */

static void impl_set_outputfile()
{
        char filename[260];

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        GetString(prhs + 2, filename, sizeof(filename));
        result = set_outputfile(lp0, (*filename) ? filename : NULL);
	*pr = (double) result;
}


/* xxlpsolve('set_pivoting', lp_handle, pivoting) */

static void impl_set_pivoting()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_pivoting(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_preferdual', lp_handle, dodual) */

static void impl_set_preferdual()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_preferdual(lp0, (MYBOOL) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_presolve', lp_handle, do_presolve) */

static void impl_set_presolve()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_presolve(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_print_sol', lp_handle, print_sol) */

static void impl_set_print_sol()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_print_sol(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_rh', lp_handle, row, value) */
/* return = xxlpsolve('set_rh', lp_handle, [values]) */

static void impl_set_rh()
{
        if (nrhs == 1+2) {
                int i, m;
                REAL *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                m = get_Nrows(lp0);
                vec = (REAL *) matCalloc(1 + m, sizeof(*vec));
        	GetRealVector(prhs + 2, vec, 0, 1 + m, TRUE);
                result = TRUE;
                for (i = 0; (i <= m) && (result); i++)
                        result = set_rh(lp0, i, vec[i]);
                matFree(vec);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_rh(lp0, (int) GetRealScalar(prhs + 2), GetRealScalar(prhs + 3));
        }
	*pr = (double) result;
}


/* return = xxlpsolve('set_rh_range', lp_handle, row, deltavalue) */
/* return = xxlpsolve('set_rh_range', lp_handle, [deltavalues]) */

static void impl_set_rh_range()
{
        if (nrhs == 1+2) {
                int i, m;
                REAL *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                m = get_Nrows(lp0);
                vec = (REAL *) matCalloc(1 + m, sizeof(*vec));
        	GetRealVector(prhs + 2, vec, 0, 1 + m, TRUE);
                result = TRUE;
                for (i = 0; (i < m) && (result); i++)
                        result = set_rh_range(lp0, i + 1, vec[i]);
                matFree(vec);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_rh_range(lp0, (int) GetRealScalar(prhs + 2), GetRealScalar(prhs + 3));
        }
        *pr = (double) result;
}


/* xxlpsolve('set_rh_vec', lp_handle, [rh]) */

static void impl_set_rh_vec()
{
        int m;
	REAL	*vec;

        PrepareCall(cmd, 2, nrhs, NULL);
        m = get_Nrows(lp0);
	vec = (REAL *) matCalloc(1 + m, sizeof(REAL));
	GetRealVector(prhs + 2, vec, 1, m, TRUE);
	set_rh_vec(lp0, vec);
        matFree(vec);
}


/* since always a matrix vector is provided to both set_row and set_rowex, always call the more
   performant sparse version of the two routines */
/* return = xxlpsolve('set_row', lp_handle, row_no, [row]) */
/* return = xxlpsolve('set_rowex', lp_handle, row_no, [row]) */

static void impl_set_row()
{
        int n, count;
        int	*index;
	REAL	*vec;

        pr = PrepareCall(cmd, 3, nrhs, plhs);
        n = get_Ncolumns(lp0);
        vec = (REAL *) matCalloc(1 + n, sizeof(*vec));
        index = (int *) matCalloc(1 + n, sizeof(*index));
	count = GetRealSparseVector(prhs + 3, vec, index, 1, n, 0);
	result = set_rowex(lp0, (int) GetRealScalar(prhs + 2), count, vec, index);
        *pr = (double) result;
        matFree(index);
        matFree(vec);
}


/* return = xxlpsolve('set_row_name', lp_handle, row, name) */
/* return = xxlpsolve('set_row_name', lp_handle, [names]) */

static void impl_set_row_name()
{
        if (nrhs == 1+2) {
                int m, i;
                strArray pa;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                m = get_Nrows(lp0);
                pa = GetCellCharItems(prhs + 2, m);
                result = TRUE;
                for (i = 0; (i < m) && (result); i++) {
                	GetCellString(pa + i, buf, bufsz);
                        result = set_row_name(lp0, i + 1, buf);
                }
                FreeCellCharItems(pa, m);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
	        GetString(prhs + 3, buf, bufsz);
	        result = set_row_name(lp0, (int) GetRealScalar(prhs + 2), buf);
        }
	*pr = (double) result;
}


/* xxlpsolve('set_scalelimit', lp_handle, scalelimit) */

static void impl_set_scalelimit()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_scalelimit(lp0, GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_scaling', lp_handle, scalemode) */

static void impl_set_scaling()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_scaling(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_semicont', lp_handle, column, must_be_sc) */
/* return = xxlpsolve('set_semicont', lp_handle, [must_be_sc]) */

static void impl_set_semicont()
{
        if (nrhs == 1+2) {
                int i, n, *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                n = get_Ncolumns(lp0);
                vec = (int *) matCalloc(n, sizeof(*vec));
        	GetIntVector(prhs + 2, vec, 0, n, TRUE);
                result = TRUE;
                for (i = 0; (i < n) && (result); i++)
                        result = set_semicont(lp0, i + 1, (MYBOOL) vec[i]);
                matFree(vec);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_semicont(lp0, (int) GetRealScalar(prhs + 2), (MYBOOL) GetRealScalar(prhs + 3));
        }
        *pr = (double) result;
}


/* xxlpsolve('set_sense', lp_handle, maximize) */

static void impl_set_sense()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_sense(lp0, (MYBOOL) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_simplextype', lp_handle, simplextype) */

static void impl_set_simplextype()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_simplextype(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_solutionlimit', lp_handle, simplextype) */

static void impl_set_solutionlimit()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_solutionlimit(lp0, (int) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_timeout', lp_handle, sectimeout) */

static void impl_set_timeout()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_timeout(lp0, (long) GetRealScalar(prhs + 2));
}


/* xxlpsolve('set_trace', lp_handle, trace) */

static void impl_set_trace()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_trace(lp0, (MYBOOL) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_upbo', lp_handle, column, value) */
/* return = xxlpsolve('set_upbo', lp_handle, [values]) */

static void impl_set_upbo()
{
        if (nrhs == 1+2) {
                int i, n;
                REAL *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                n = get_Ncolumns(lp0);
                vec = (REAL *) matCalloc(n, sizeof(*vec));
        	GetRealVector(prhs + 2, vec, 0, n, TRUE);
                result = TRUE;
                for (i = 0; (i < n) && (result); i++)
                        result = set_upbo(lp0, i + 1, vec[i]);
                matFree(vec);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_upbo(lp0, (int) GetRealScalar(prhs + 2), GetRealScalar(prhs + 3));
        }
        *pr = (double) result;
}


/* return = xxlpsolve('set_var_branch', lp_handle, column, branch_mode) */
/* return = xxlpsolve('set_var_branch', lp_handle, [branch_mode]) */

static void impl_set_var_branch()
{
        if (nrhs == 1+2) {
                int i, n, *vec;

                pr = PrepareCall(cmd, 2, nrhs, plhs);
                n = get_Ncolumns(lp0);
                vec = (int *) matCalloc(n, sizeof(*vec));
        	GetIntVector(prhs + 2, vec, 0, n, TRUE);
                result = TRUE;
                for (i = 0; (i < n) && (result); i++)
                        result = set_var_branch(lp0, i + 1, vec[i]);
                matFree(vec);
        }
        else {
	        pr = PrepareCall(cmd, 3, nrhs, plhs);
		result = set_var_branch(lp0, (int) GetRealScalar(prhs + 2), (int) GetRealScalar(prhs + 3));
        }
        *pr = (double) result;
}


/* return = xxlpsolve('set_var_weights', lp_handle, [weights]) */

static void impl_set_var_weights()
{
        int n;
	REAL	*vec;

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        n = get_Ncolumns(lp0);
	vec = (REAL *) matCalloc(n, sizeof(REAL));
	GetRealVector(prhs + 2, vec, 0, n, TRUE);
	result = set_var_weights(lp0, vec);
        *pr = (double) result;
        matFree(vec);
}


/* xxlpsolve('set_verbose', lp_handle, verbose) */

static void impl_set_verbose()
{
        PrepareCall(cmd, 2, nrhs, NULL);
	set_verbose(lp0, (int) GetRealScalar(prhs + 2));
}


/* return = xxlpsolve('set_XLI', lp_handle, filename) */

static void impl_set_XLI()
{
        char filename[260];

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        GetString(prhs + 2, filename, sizeof(filename));
        result = set_XLI(lp0, filename);
	*pr = (double) result;
}


/* result = xxlpsolve('solve', lp_handle) */

static void impl_solve()
{
        /* int m, n, i */;

        pr = PrepareCall(cmd, 1, nrhs, plhs);
	result = solve(lp0);
	*pr = (double) result;
        switch (result) {
        case OPTIMAL:
        case SUBOPTIMAL:
        case PROCBREAK:
        case FEASFOUND:
/*
          if (get_verbose(lp0) >= DETAILED) {
	    Printf("Branch & Bound depth: %d\n", get_max_level(lp0));
	    Printf("Nodes processed: %d\n", get_total_nodes(lp0));
	    Printf("Simplex pivots: %d\n", get_total_iter(lp0));
            Printf("Number of equal solutions: %d\n\n", get_solutioncount(lp0));
          }
          if (get_verbose(lp0) >= NORMAL) {
            Printf("Value of objective function: %f\n\n", get_objective(lp0));
            Printf("Actual values of the variables:\n");
            m = get_Nrows(lp0);
            n = get_Ncolumns(lp0);
            for (i = 1; i <= n; i++)
              Printf("%s: %f\n", get_col_name(lp0, i), get_var_primalresult(lp0, m + i));
          }
*/
          break;
        case NOMEMORY:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("Out of memory\n");
          break;
        case INFEASIBLE:
          if (get_verbose(lp0) >= NORMAL)
	  	Printf("This problem is infeasible\n");
          break;
        case UNBOUNDED:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("This problem is unbounded\n");
          break;
        case PROCFAIL:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("The B&B routine failed\n");
          break;
        case TIMEOUT:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("Timeout\n");
          break;
        case USERABORT:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("User aborted\n");
          break;
        case DEGENERATE:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("This problem is degenerative\n");
          break;
        case NUMFAILURE:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("Numerical failure encountered\n");
          break;
        case NOFEASFOUND:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("No feasible branch and bound solution found\n");
          break;
        default:
          if (get_verbose(lp0) >= NORMAL)
          	Printf("lp_solve failed\n");
          break;
        }
}


/* return = xxlpsolve('time_elapsed', lp_handle) */

static void impl_time_elapsed()
{
        pr = PrepareCall(cmd, 1, nrhs, plhs);
        *pr = time_elapsed(lp0);
}


/* xxlpsolve('unscale', lp_handle) */

static void impl_unscale()
{
        PrepareCall(cmd, 1, nrhs, NULL);
        unscale(lp0);
}


/* return = xxlpsolve('write_freemps', lp_handle, filename) */
/* return = xxlpsolve('write_freeMPS', lp_handle, filename) */

static void impl_write_freemps()
{
        char filename[260];

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        GetString(prhs + 2, filename, sizeof(filename));
        result = write_freemps(lp0, filename);
	*pr = (double) result;
}


/* return = xxlpsolve('write_lp', lp_handle, filename) */
/* return = xxlpsolve('write_LP', lp_handle, filename) */

static void impl_write_lp()
{
        char filename[260];

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        GetString(prhs + 2, filename, sizeof(filename));
        result = write_lp(lp0, filename);
	*pr = (double) result;
}


/* return = xxlpsolve('write_mps', lp_handle, filename) */
/* return = xxlpsolve('write_MPS', lp_handle, filename) */

static void impl_write_mps()
{
        char filename[260];

        pr = PrepareCall(cmd, 2, nrhs, plhs);
        GetString(prhs + 2, filename, sizeof(filename));
        result = write_mps(lp0, filename);
	*pr = (double) result;
}


/* return = xxlpsolve('write_XLI', lp_handle, filename {, options {, results}}) */

static void impl_write_XLI()
{
        char filename[260], options[50];
        int n;
        MYBOOL results;

        if (nrhs == 1+2)
                n = 2;
        else if (nrhs == 1+3)
                n = 3;
        else
                n = 4;

        pr = PrepareCall(cmd, n, nrhs, plhs);
        GetString(prhs + 2, filename, sizeof(filename));
        if (n >= 3)
        	GetString(prhs + 3, options, sizeof(options));
        else
                *options = 0;
        if (n >= 4)
                results = (MYBOOL) GetRealScalar(prhs + 4);
        else
                results = FALSE;
        result = write_XLI(lp0, filename, options, results);
	*pr = (double) result;
}


static struct {
        char *cmd;
        impl_routine *routine;
        int needshandle;
} routines[] = {
  { "add_column", impl_add_column, TRUE },
  { "add_columnex", impl_add_column, TRUE },
  { "add_constraint", impl_add_constraint, TRUE },
  { "add_constraintex", impl_add_constraint, TRUE },
  { "add_SOS", impl_add_SOS, TRUE },
  { "column_in_lp", impl_column_in_lp, TRUE },
  { "default_basis", impl_default_basis, TRUE },
  { "del_column", impl_del_column, TRUE },
  { "del_constraint", impl_del_constraint, TRUE },
  { "delete_lp", impl_delete_lp, TRUE },
  { "free_lp", impl_delete_lp, TRUE },
  { "get_anti_degen", impl_get_anti_degen, TRUE },
  { "get_basis", impl_get_basis, TRUE },
  { "get_basiscrash", impl_get_basiscrash, TRUE },
  { "get_bb_depthlimit", impl_get_bb_depthlimit, TRUE },
  { "get_bb_floorfirst", impl_get_bb_floorfirst, TRUE },
  { "get_bb_rule", impl_get_bb_rule, TRUE },
  { "get_bounds_tighter", impl_get_bounds_tighter, TRUE },
  { "get_break_at_value", impl_get_break_at_value, TRUE },
  { "get_col_name", impl_get_col_name, TRUE },
  { "get_column", impl_get_column, TRUE },
  { "get_constr_type", impl_get_constr_type, TRUE },
  { "get_constraints", impl_get_constraints, TRUE },
  { "get_dual_solution", impl_get_dual_solution, TRUE },
  { "get_epsb", impl_get_epsb, TRUE },
  { "get_epsd", impl_get_epsd, TRUE },
  { "get_epsel", impl_get_epsel, TRUE },
  { "get_epsint", impl_get_epsint, TRUE },
  { "get_epsperturb", impl_get_epsperturb, TRUE },
  { "get_epspivot", impl_get_epspivot, TRUE },
  { "get_improve", impl_get_improve, TRUE },
  { "get_infinite", impl_get_infinite, TRUE },
  { "get_lowbo", impl_get_lowbo, TRUE },
  { "get_lp_index", impl_get_lp_index, TRUE },
  { "get_lp_name", impl_get_lp_name, TRUE },
  { "get_mat", impl_get_mat, TRUE },
  { "get_max_level", impl_get_max_level, TRUE },
  { "get_maxpivot", impl_get_maxpivot, TRUE },
  { "get_mip_gap", impl_get_mip_gap, TRUE },
  { "get_nameindex", impl_get_nameindex, TRUE },
  { "get_Ncolumns", impl_get_Ncolumns, TRUE },
  { "get_negrange", impl_get_negrange, TRUE },
  { "get_nonzeros", impl_get_nonzeros, TRUE },
  { "get_Norig_columns", impl_get_Norig_columns, TRUE },
  { "get_Norig_rows", impl_get_Norig_rows, TRUE },
  { "get_Nrows", impl_get_Nrows, TRUE },
  { "get_obj_bound", impl_get_obj_bound, TRUE },
  { "get_objective", impl_get_objective, TRUE },
  { "get_orig_index", impl_get_orig_index, TRUE },
  { "get_origcol_name", impl_get_origcol_name, TRUE },
  { "get_origrow_name", impl_get_origrow_name, TRUE },
  { "get_pivoting", impl_get_pivoting, TRUE },
  { "get_presolve", impl_get_presolve, TRUE },
  { "get_primal_solution", impl_get_primal_solution, TRUE },
  { "get_print_sol", impl_get_print_sol, TRUE },
  { "get_rh", impl_get_rh, TRUE },
  { "get_rh_range", impl_get_rh_range, TRUE },
  { "get_row", impl_get_row, TRUE },
  { "get_row_name", impl_get_row_name, TRUE },
  { "get_scalelimit", impl_get_scalelimit, TRUE },
  { "get_scaling", impl_get_scaling, TRUE },
  { "get_sensitivity_obj", impl_get_sensitivity_objex, TRUE },
  { "get_sensitivity_objex", impl_get_sensitivity_objex, TRUE },
  { "get_sensitivity_rhs", impl_get_sensitivity_rhsex, TRUE },
  { "get_sensitivity_rhsex", impl_get_sensitivity_rhsex, TRUE },
  { "get_simplextype", impl_get_simplextype, TRUE },
  { "get_solution", impl_get_solution, TRUE },
  { "get_solutioncount", impl_get_solutioncount, TRUE },
  { "get_solutionlimit", impl_get_solutionlimit, TRUE },
  { "get_status", impl_get_status, TRUE },
  { "get_statustext", impl_get_statustext, TRUE },
  { "get_timeout", impl_get_timeout, TRUE },
  { "get_total_iter", impl_get_total_iter, TRUE },
  { "get_total_nodes", impl_get_total_nodes, TRUE },
  { "get_upbo", impl_get_upbo, TRUE },
  { "get_var_branch", impl_get_var_branch, TRUE },
  { "get_var_dualresult", impl_get_var_dualresult, TRUE },
  { "get_var_primalresult", impl_get_var_primalresult, TRUE },
  { "get_var_priority", impl_get_var_priority, TRUE },
  { "get_variables", impl_get_variables, TRUE },
  { "get_verbose", impl_get_verbose, TRUE },
  { "get_working_objective", impl_get_working_objective, TRUE },
  { "has_BFP", impl_has_BFP, TRUE },
  { "has_XLI", impl_has_XLI, TRUE },
  { "is_add_rowmode", impl_is_add_rowmode, TRUE },
  { "is_anti_degen", impl_is_anti_degen, TRUE },
  { "is_binary", impl_is_binary, TRUE },
  { "is_break_at_first", impl_is_break_at_first, TRUE },
  { "is_constr_type", impl_is_constr_type, TRUE },
  { "is_debug", impl_is_debug, TRUE },
  { "is_feasible", impl_is_feasible, TRUE },
  { "is_free", impl_is_free, TRUE },
  { "is_infinite", impl_is_infinite, TRUE },
  { "is_int", impl_is_int, TRUE },
  { "is_integerscaling", impl_is_integerscaling, TRUE },
  { "is_maxim", impl_is_maxim, TRUE },
  { "is_nativeBFP", impl_is_nativeBFP, TRUE },
  { "is_nativeXLI", impl_is_nativeXLI, TRUE },
  { "is_negative", impl_is_negative, TRUE },
  { "is_piv_mode", impl_is_piv_mode, TRUE },
  { "is_piv_rule", impl_is_piv_rule, TRUE },
  { "is_presolve", impl_is_presolve, TRUE },
  { "is_scalemode", impl_is_scalemode, TRUE },
  { "is_scaletype", impl_is_scaletype, TRUE },
  { "is_semicont", impl_is_semicont, TRUE },
  { "is_SOS_var", impl_is_SOS_var, TRUE },
  { "is_trace", impl_is_trace, TRUE },
  { "lp_solve_version", impl_lp_solve_version, FALSE },
  { "make_lp", impl_make_lp, FALSE },
  { "print_constraints", impl_print_constraints, TRUE },
  { "print_debugdump", impl_print_debugdump, TRUE },
  { "print_duals", impl_print_duals, TRUE },
  { "print_lp", impl_print_lp, TRUE },
  { "print_objective", impl_print_objective, TRUE },
  { "print_scales", impl_print_scales, TRUE },
  { "print_solution", impl_print_solution, TRUE },
  { "print_str", impl_print_str, TRUE },
  { "print_tableau", impl_print_tableau, TRUE },
  { "read_freemps", impl_read_freeMPS, FALSE },
  { "read_freeMPS", impl_read_freeMPS, FALSE },
  { "read_lp", impl_read_LP, FALSE },
  { "read_LP", impl_read_LP, FALSE },
  { "read_mps", impl_read_MPS, FALSE },
  { "read_MPS", impl_read_MPS, FALSE },
  { "read_XLI", impl_read_XLI, FALSE },
  { "set_add_rowmode", impl_set_add_rowmode, TRUE },
  { "set_anti_degen", impl_set_anti_degen, TRUE },
  { "set_basis", impl_set_basis, TRUE },
  { "set_basiscrash", impl_set_basiscrash, TRUE },
  { "set_bb_depthlimit", impl_set_bb_depthlimit, TRUE },
  { "set_bb_floorfirst", impl_set_bb_floorfirst, TRUE },
  { "set_bb_rule", impl_set_bb_rule, TRUE },
  { "set_BFP", impl_set_BFP, TRUE },
  { "set_binary", impl_set_binary, TRUE },
  { "set_bounds", impl_set_bounds, TRUE },
  { "set_bounds_tighter", impl_set_bounds_tighter, TRUE },
  { "set_break_at_first", impl_set_break_at_first, TRUE },
  { "set_break_at_value", impl_set_break_at_value, TRUE },
  { "set_col_name", impl_set_col_name, TRUE },
  { "set_column", impl_set_column, TRUE },
  { "set_columnex", impl_set_column, TRUE },
  { "set_constr_type", impl_set_constr_type, TRUE },
  { "set_debug", impl_set_debug, TRUE },
  { "set_epsb", impl_set_epsb, TRUE },
  { "set_epsd", impl_set_epsd, TRUE },
  { "set_epsel", impl_set_epsel, TRUE },
  { "set_epsint", impl_set_epsint, TRUE },
  { "set_epsperturb", impl_set_epsperturb, TRUE },
  { "set_epspivot", impl_set_epspivot, TRUE },
  { "set_free", impl_set_free, TRUE },
  { "set_improve", impl_set_improve, TRUE },
  { "set_infinite", impl_set_infinite, TRUE },
  { "set_int", impl_set_int, TRUE },
  { "set_lowbo", impl_set_lowbo, TRUE },
  { "set_lp_name", impl_set_lp_name, TRUE },
  { "set_mat", impl_set_mat, TRUE },
  { "set_maxim", impl_set_maxim, TRUE },
  { "set_maxpivot", impl_set_maxpivot, TRUE },
  { "set_minim", impl_set_minim, TRUE },
  { "set_mip_gap", impl_set_mip_gap, TRUE },
  { "set_negrange", impl_set_negrange, TRUE },
  { "set_obj", impl_set_obj, TRUE },
  { "set_obj_bound", impl_set_obj_bound, TRUE },
  { "set_obj_fn", impl_set_obj_fn, TRUE },
  { "set_obj_fnex", impl_set_obj_fn, TRUE },
  { "set_outputfile", impl_set_outputfile, TRUE },
  { "set_pivoting", impl_set_pivoting, TRUE },
  { "set_preferdual", impl_set_preferdual, TRUE },
  { "set_presolve", impl_set_presolve, TRUE },
  { "set_print_sol", impl_set_print_sol, TRUE },
  { "set_rh", impl_set_rh, TRUE },
  { "set_rh_range", impl_set_rh_range, TRUE },
  { "set_rh_vec", impl_set_rh_vec, TRUE },
  { "set_row", impl_set_row, TRUE },
  { "set_rowex", impl_set_row, TRUE },
  { "set_row_name", impl_set_row_name, TRUE },
  { "set_scalelimit", impl_set_scalelimit, TRUE },
  { "set_scaling", impl_set_scaling, TRUE },
  { "set_semicont", impl_set_semicont, TRUE },
  { "set_sense", impl_set_sense, TRUE },
  { "set_simplextype", impl_set_simplextype, TRUE },
  { "set_solutionlimit", impl_set_solutionlimit, TRUE },
  { "set_timeout", impl_set_timeout, TRUE },
  { "set_trace", impl_set_trace, TRUE },
  { "set_upbo", impl_set_upbo, TRUE },
  { "set_var_branch", impl_set_var_branch, TRUE },
  { "set_var_weights", impl_set_var_weights, TRUE },
  { "set_verbose", impl_set_verbose, TRUE },
  { "set_XLI", impl_set_XLI, TRUE },
  { "solve", impl_solve, TRUE },
  { "time_elapsed", impl_time_elapsed, TRUE },
  { "unscale", impl_unscale, TRUE },
  { "write_freemps", impl_write_freemps, TRUE },
  { "write_freeMPS", impl_write_freemps, TRUE },
  { "write_lp", impl_write_lp, TRUE },
  { "write_LP", impl_write_lp, TRUE },
  { "write_mps", impl_write_mps, TRUE },
  { "write_MPS", impl_write_mps, TRUE },
  { "write_XLI", impl_write_XLI, TRUE },

/* extra routines */

#if defined DEBUG || defined DEMO
  { "demo", impl_demo, FALSE },
#endif
  { "get_col_names", impl_get_col_name, TRUE },
  { "get_constr_types", impl_get_constr_type, TRUE },
  { "get_int", impl_is_int, TRUE },
  { "get_no_cols", impl_get_Ncolumns, TRUE },
  { "get_no_rows", impl_get_Nrows, TRUE },
  { "get_objective_name", impl_get_objective_name, TRUE },
  { "get_obj_fn", impl_get_obj_fn, TRUE },
  { "get_obj_fun", impl_get_obj_fn, TRUE },
  { "get_problem_name", impl_get_lp_name, TRUE },
  { "get_reduced_costs", impl_get_dual_solution, TRUE },
  { "get_row_names", impl_get_row_name, TRUE },
  { "mat_elm", impl_get_mat, TRUE },
  { "print_handle", impl_print_handle, FALSE },
  { "read_lp_file", impl_read_LP, FALSE },
};

callerPrototype
{
	int	i;
        hashelem *hp;

        publicargs();

	if (!initialized) {
		/* Register the Exit Function */
                registerExitFcn();

		/* Allocate a string array to store command */

		cmd = (char *) calloc(cmdsz, sizeof(*cmd));

		/* Allocate a string array to store error message */

		errmsg = (char *) calloc(errmsgsz, sizeof(*errmsg));

                /* create hashtable of all callbable commands to find them back quickly */
                hashtab = create_hash_table(sizeof(routines) / sizeof(*routines), 0);
                for (i = 0; i < sizeof(routines)/sizeof(*routines); i++)
                  	puthash(routines[i].cmd, i, NULL, hashtab);

		/* Initialise the lp array, and pointer to the last lp */

		lp_last = -1;

                if (!init_lpsolve_lib())
                        ErrMsgTxt("Failed to initialise lpsolve library.\n");

        	initialized = TRUE;
#               if defined DEBUG
                	Printf("Initialised\n");
#               endif
	}

	/* Get the first argument as a string matrix */

	if (nrhs < 1) {
		int majorversion, minorversion, release, build;

		/* ErrMsgTxt("At least one command is required."); */
                lp_solve_version(&majorversion, &minorversion, &release, &build);
                Printf(drivername "  " caller " Interface version " driverVERSION "\n" \
                       "using lpsolve version %d.%d.%d.%d\n\n" \
                       "Usage: [ret1, ret2, ...] = " drivername "('functionname', arg1, arg2, ...)\n",
                       majorversion, minorversion, release, build);
                return;
        }

	GetString(prhs, cmd, cmdsz);

#       if defined DEBUG
       		Printf("%s\n", cmd);
#       endif

	/* Now call the required part of the lp toolkit */

/*
        for (i = 0; (i < sizeof(routines)/sizeof(*routines)) && (strcmp(routines[i].cmd, cmd)); i++);
        if (i >= sizeof(routines) / sizeof(*routines)) {
                strcpy(errmsg, cmd);
                strncat(errmsg,": Unimplemented.", errmsgsz);
                ErrMsgTxt(errmsg);
        }
*/

        hp = findhash(cmd, hashtab);
        if (hp == NULL) {
                strcpy(errmsg, cmd);
                strncat(errmsg,": Unimplemented.", errmsgsz);
                ErrMsgTxt(errmsg);
        }
        i = hp->index;

        if (routines[i].needshandle) {
                if (nrhs < 2)
		        ErrMsgTxt("An lp handle is required.");

        	if ((!handle_valid(h = (int) GetRealScalar(&prhs[1]))) || ((lp0 = lp[h]) == NULL)) {
                	strcpy(errmsg, cmd);
                	strncat(errmsg, ": Invalid lp handle.", errmsgsz);
        		ErrMsgTxt(errmsg);
        	}
        }
        routines[i].routine();
}
