% Demonstrates the GTM with a 2D target space and a 1D latent space.
%
%		This script generates a simple data set in 2 dimensions, 
%		with an intrinsic dimensionality of 1, and trains a GTM 
%		with a 1-dimensional latent variable to model this data 
%		set, visually illustrating the training process
%
% Synopsis:	gtm_demo
%
% Notes:	The script generates a number of variables which may
%		overwrite variables already existing in the workspace.
%		The generated variables remain in the work space after
%		the script has finished executing.
%

% Version:	The GTM Toolbox v1.0 beta
%
% Copyright:	The GTM Toolbox is distributed under the GNU General Public 
%		Licence (version 2 or later); please refer to the file 
%		licence.txt, included with the GTM Toolbox, for details.
%
%		(C) Copyright Markus Svensen, 1996


%%%%% Generate and plot a 2D data set %%%%%

fprintf(['\nYou''ve started the GTM demo, please wait while ', ...
	 'data is being generated.\n\n']);
T = [0.15:0.05:3.05]';
T = [T (T + 1.25*sin(2*T))];
clf;
plot(T(:,1), T(:,2), 'ro');
axis([0 3.5 0 3.5]);
set(gca,'XTickLabels',[],'YTickLabels',[]);
fprintf([...
'The figure shows data generated by feeding a 1D uniform distribution\n', ...
'(on the X-axis) through a non-linear function (y = x + 1.25*sin(2*x))\n', ...
'\nPress any key to continue ...\n\n']);
pause;



%%%%% Generate a unit circle figure, to be used for plotting %%%%%

src = [0:(2*pi)/(20-1):2*pi]';
unitC = [sin(src) cos(src)];



%%%%% Generate and plot (along with the data) an initial GTM model %%%%%

fprintf('Please wait while the GTM model is set up.\n\n');
[X, MU, FI, W, b] = gtm_stp1(T, 20, 5, 2);
Y =  FI*W;
hold off;
plot(Y(:,1),  Y(:,2), 'g');
hold on;
for i=1:20
  c = 2*unitC*sqrt(1/b) + [ones(20,1)*Y(i,1) ones(20,1)*Y(i,2)];
  fill(c(:,1), c(:,2), [0.8 1 0.8]);
end
plot(T(:,1), T(:,2), 'ro');
plot(Y(:,1),  Y(:,2), 'g+');
plot(Y(:,1),  Y(:,2), 'g');
axis([0 3.5 0 3.5]);
set(gca,'XTickLabels',[],'YTickLabels',[]);
drawnow;
title('Initial configuration');
fprintf([...
'The figure shows the starting point for the GTM, before the training.\n', ...
'A discrete latent variable distribution of 20 points in 1 dimension \n', ...
'is mapped to the 1st principal component of the target data.\n', ...
'Each of the 20 points defines the centre of a Gaussian in a Gaussian \n', ...
'mixture, marked by the green ''+''-signs. The mixture components have \n', ...
'all equal variance, illustrated by the filled circle around each \n', ...
'''+''-sign, the raddii corresponding to 2 standard deviations.\n', ...
'The ''+''-signs are connected with a line according to their \n', ...
'corresponding ordering in latent space.\n\n', ...
'Press any key to begin training ...\n\n']);
pause;



%%%% Train the GTM and plot it (along with the data) as training proceeds %%%%

for j = 1:15
  [W, b] = gtm_trn(T, FI, W, 0.0, 1, b, 'quiet');
  hold off;
  Y =  FI*W;
  plot(Y(:,1),  Y(:,2), 'g');
  hold on;
  for i=1:20
    c = 2*unitC*sqrt(1/b) + [ones(20,1)*Y(i,1) ones(20,1)*Y(i,2)];
    fill(c(:,1), c(:,2), [0.8 1.0 0.8]);
  end
  plot(T(:,1), T(:,2), 'ro');
  plot(Y(:,1),  Y(:,2), 'g+');
  plot(Y(:,1),  Y(:,2), 'g');
  axis([0 3.5 0 3.5]);
  set(gca,'XTickLabels',[],'YTickLabels',[]);
  title(['After ', int2str(j),' iterations of training.']);
  drawnow;
  if (j == 4)
    fprintf([...
'The GTM initiallaly adapts relatively quickly - already after \n', ...
'4 iterations of training, a rough fit is attained.\n\n', ...
'Press any key to continue training ...\n\n']);
    pause;
  elseif (j == 8)
    fprintf([...
'After another 4 iterations of training:  from now on further \n', ...
'training only makes small changes to the mapping, which combined with \n', ...
'decrements of the Gaussian mixture variance, optimize the fit in \n', ...
'terms of likelihood.\n\n', ...
'Press any key to continue training ...\n\n']);
    pause;
  else
    pause(1);
  end
end


fprintf([...
'After 15 iterations of training the GTM can be regarded as converged. \n', ...
'Is has been adapted to fit the target data distribution as well \n', ...
'as possible, given prior smoothness constraints on the mapping. It \n', ...
'captures the fact that the probabilty density is higher at the two \n', ...
'bends of the curve, and lower towards its end points.\n\n', ...
'Thanks for your interest in running the GTM demo.\n\n']);





