function DrawTest
% DrawTest is a two part test, printing in the Command Window a record of
% an objective test of several drawing functions, and displaying to the
% user (for 1 sec) an image resulting from exercising all the drawing
% functions. This is done for each screen and pixelsize, both for a window
% and an offscreenWindow.
% 
% The objective test assesses whether various Screen drawing operations
% actually poke the right color into the window (or offscreen window). We
% test Screen FillRect, DrawText, FrameRect, DrawLine, PutImage, and
% CopyWindow. We use Screen GetImage to read back the pixels and check
% that they have the same numerical value as the "color" argument supplied
% to the drawing command.
% 
% If Screen fails this test in Psychtoolbox version 2.5 or later, please
% report it to the forum,
% web mailto:psychtoolbox@yahoogroups.com
% along with a report from ScreenTest.
% 
% This test was created in response to the 10/6/00 bug report by Erin Harley, 
% eharley@u.washington.edu
% http://groups.yahoo.com/group/psychtoolbox/message/266
% http://groups.yahoo.com/group/psychtoolbox/message/881
% That bug is fixed in Psychtoolbox version 2.5.

% 4/1/98 dgp Wrote subjective test, as ColorTest.
% 4/6/98 dgp Wait 2 s instead of waiting for click.
% 4/1/02 dgp Renamed DrawTest. Added objective test, based on 10/6/00 bug report 
%            by Erin Harley, eharley@u.washington.edu
% 4/4/02 dgp Cosmetic enhancements.
% 4/24/02 awi Exit on PC with message.

if strcmp('PCWIN',computer)
    error('Win: DrawTest.m not yet supported.');
end


fprintf('DrawTest\n');
DescribeComputer;
for whichScreen=Screen('Screens')
	fprintf('*** Screen %d *************************************************************\n',whichScreen);
	for pixelSize=Screen(whichScreen,'PixelSizes');
		fprintf('%d bit pixelSize\n',pixelSize);
		window=Screen(whichScreen,'OpenWindow',[],[],pixelSize); 
		windowRect=Screen(window,'Rect');
		windowColor=WhiteIndex(whichScreen);
		textColor=round(GrayIndex(whichScreen,0.4));
		maxColor=max(BlackIndex(whichScreen),WhiteIndex(whichScreen));
		oWindow=Screen(window,'OpenOffscreenWindow',windowColor);
		for w=[window oWindow]
			if w==window
				fprintf('  Window\n');
			else
				fprintf('  Offscreen window\n');
				
				% test CopyWindow
				Screen('CopyWindow',window,oWindow);
				original=Screen(window,'GetImage',[0 0 256 10]);
				copy=Screen(oWindow,'GetImage',[0 0 256 10]);
				if all(original==copy)
					fprintf('    ok CopyWindow\n');
				else
					fprintf('    CopyWindow color error\n');
				end
			 	Screen(w,'FillRect',windowColor);
			end
		
			% do a write-then-read test of several drawing functions
			ramp=0:maxColor;
			Screen(w,'PutImage',ramp,[0 0 length(ramp) 1]);
			getRamp=Screen(w,'GetImage',[0 0 length(ramp) 1]);
			if all(getRamp(1,:,1)==ramp)
				fprintf('    ok PutImage\n');
			else
				fprintf('    PutImage color error\n');
			end
			Screen(w,'PutImage',windowColor,windowRect);

			Screen(w,'DrawText','g',0,10,textColor);
			imageArray=Screen(w,'GetImage',[0 0 256 10]);
			if isempty(setxor(double(imageArray(:)),[windowColor textColor]'))
				fprintf('    ok DrawText\n');
			else
				fprintf('    DrawText color error or text smoothing is on in Control Panel Appearance or ~ATM.\n');
			end
			Screen(w,'PutImage',windowColor,windowRect);
			
		 	Screen(w,'FrameRect',textColor,[2 2 4 4],2,2);
		 	Screen(w,'DrawLine',textColor,0,0,10,10,2,2); 
			imageArray=Screen(w,'GetImage',[0 0 256 10]);
			if isempty(setxor(double(imageArray(:)),[windowColor textColor]'))
				fprintf('    ok FrameRect and DrawLine\n');
			else
				fprintf('    FrameRect or DrawLine color error\n');
			end
			Screen(w,'PutImage',windowColor,windowRect);

			colorErrors=0;
			for color=0:maxColor
			 	Screen(w,'FillRect',color);
				getColor=double(Screen(w,'GetImage',[0 0 1 1]));
				if getColor(1)~=color
					colorErrors=colorErrors+1;
					inOut(colorErrors,1)=color;
					inOut(colorErrors,2)=getColor(1);
				end
			end
			if colorErrors
				fprintf('    FillRect %d color errors. (Showing first 12.)\n',colorErrors);
				format compact
				disp(inOut(1:12,:,:)');
			else
				fprintf('    ok FillRect\n');
			end
			Screen(w,'PutImage',windowColor,windowRect);

			
			% exercise all the drawing functions
			white=WhiteIndex(w);
			black=BlackIndex(w);
			gray=round(GrayIndex(w,0.5));
			if gray==white
				gray=black;
			end
			r=Screen(w,'Rect');
			Screen(w,'DrawLine',GrayIndex(w,0.3),0,r(RectBottom)/2,r(RectRight),r(RectBottom)/2,10,10);
			Screen(w,'FrameArc',GrayIndex(w,0.8),[],0,90,10,10);
			Screen(w,'FrameOval',GrayIndex(w,0.1),[],10,10);
			Screen(w,'FrameRect',GrayIndex(w,0.7),[],10,10);
		 	Screen(w,'FillPoly',white,[200 400;400 200;1000 400;400 600;200 400]);
			if psychtoolboxversion>2.44
			  	Screen(w,'FillArc',GrayIndex(w,0.8),[],0,90);
			end
			Screen(w,'FillOval',gray,[600 200 1000 600]);
			Screen(w,'FillRect',black,[200 200 600 600]);
			Screen(w,'TextSize',30);
			string=sprintf('pixelSize %d',pixelSize);
			if pixelSize>8
				color=[white,black,black];	% red
				string=[string ', RGB color'];
			else
				color=gray;
				string=[string ', grayscale'];
			end
			Screen(w,'DrawText',string,250,412,color);
			ramp=0:max(white,black);
			Screen(w,'PutImage',ramp,[0 0 length(ramp) length(ramp)]);
			if w==window
				WaitSecs(1); % pause to allow user to admire the drawing
			end
		end
		Screen(w,'Close');
		Screen(window,'Close');
	end
end
fprintf('**************************************************************************\n')
