function ScreenSaverTest
% ScreenSaverTest tests the following 3 actions:
% 1. ScreenSaver.mex enables/disables any AfterDark-compatible screen saver
% and Energy Saver screen dimming. (See ScreenSaver for explanation.)
% 2. Screen.mex automatically disables screen saving and dimming
% while it has onscreen windows open. 
% 3. Screen.mex pretends to be a sleeping screen saver. It does this to
% encourage other processes to leave the screen alone.
% E.g. this is moderately effective in causing Drop Drawers to hide its 
% system-wide floating windows.
% web http://www.sigsoftware.com/dropdrawers/
%
% The test reports the bits of the Gestalt 'SAVR' selector,
% which are used by AfterDark-compatible screen savers to announce their
% current state. When Screen.mex pretends that it's a screen saver, the 
% value of the 'SAVR' selector is frozen, with the "asleep" and "enabled" 
% bits both set to 1.
%
% NOTE: If you use the screen saver Sleeper, please use version 3.2.1b6 or better,
% as in earlier versions the "Power down monitor" mode could not be enabled/disabled
% by another process.
% web http://www.stclairsoft.com/Sleeper/
% You may need to get it from their beta page:
% web http://www.stclairsoft.com/Sleeper/beta.html

% 9/28/99 dgp wrote it.
% 10/3/99 dgp modified to help the user notice whether we're succeeding
%             in hiding drop drawers.
% 12/23/99 dgp added explanatory print out.
% 1/18/00 dgp Add tests of enhancement for Energy Saver control panel.
% 1/21/00 dgp Recommend upgrading to Sleeper 3.2.1b6 or better.
% 4/24/02 awi Exit on PC with message.

if strcmp('PCWIN',computer)
    error('Win: ScreenSaverTest not yet supported.');
end

clear screen
fprintf('ScreenSaverTest.m\n');
fprintf('     By convention, most screen savers use the Gestalt ''SAVR'' selector\n');
fprintf('to announce their state, i.e. (sleep) enabled or not, and  asleep or not.\n');
fprintf('Here we check that our calls to ScreenSaver.mex have the intended effect\n');
fprintf('on the actual screen saver, by monitoring what the actual screen saver announces\n');
fprintf('in the Gestalt ''SAVR'' selector. The argument to ScreenSaver.mex should enable/disable\n');
fprintf('the screen saver. If there''s no screen saver installed then ScreenSaver.mex\n');
fprintf('will have no effect.\n');
fprintf('     We also test a new feature of Screen.mex, whereby it pretends to be a \n');
fprintf('sleeping screen saver to discourage other processes from messing with the \n');
fprintf('screen. \n\n');

fprintf('%-18s enabled asleep\n','Gestalt ''SAVR'':');
savr
screensaver(1);
savr('screensaver(1)')
screensaver(0);
savr('screensaver(0)')
r=screen(0,'rect');
r(RectLeft)=r(RectRight)-100;
gray=GrayIndex(0,0.8);
w=screen(0,'openwindow',gray,r);
pixelSize=screen(0,'pixelsize');
savr(sprintf('open window %d bit',pixelSize))
pause(1)
screen(w,'close')
savr('close')
others=setdiff(screen(0,'pixelsizes'),[screen(0,'pixelsize')]);
pixelSize=others(1);
w=screen(0,'openwindow',gray,r,pixelSize);
savr(sprintf('open window %d bit',pixelSize))
pause(1)
screen(w,'close')
savr('close')
screensaver(1);
savr('screensaver(1)')
screensaver(0);
savr('screensaver(0)')

fprintf('\n');
ScreenSaver(0,0);
[wasSaveEnabled0,wasDimEnabled0]=ScreenSaver(1,1);
[wasSaveEnabled1,wasDimEnabled1]=ScreenSaver(0,0);
if wasSaveEnabled0
	fprintf('ERROR: Screen saver cannot be disabled!\n');
end
if wasDimEnabled0
	fprintf('ERROR: Energy Saver screen dimmer cannot be disabled!\n');
end
if wasSaveEnabled1
	fprintf('SUCCESS: Screen saver present, and can be enabled and disabled.\n');
else
	fprintf('No screen saver seems to be present.\n');
end
if wasDimEnabled1
	fprintf('SUCCESS: Energy Saver screen dimming can be enabled and disabled.\n');
else
	fprintf('Energy Saver screen dimming appears not to be available.\n');
end

%

function savr(s)
if nargin<1
	s='';
end
g=gestalt('SAVR');
if length(g)>1
	fprintf(1,'%-18s %4d %6d\n',s,g(32),g(31));
else
	fprintf(1,'%-18s gestalt ''SAVR'' error %d\n',s,g);
end

