function ShowtimeLeakTest
% ShowtimeLeakTest measures any memory leakage associated with use of Showtime.mex.
%
% I found a memory leak in FPRINTF, approximately equal to the
% length of the formatted string, plus 1, from TempFreeMem().
%
% However, this program also loses about 500 bytes from FreeMem,
% which isn't recovered by the calls to CLEAR ALL. I'm not sure
% that this is a bug in Showtime.mex. It may simply reflect somewhat sloppy
% memory management by Matlab since Showtime creates Matlab variables that
% it returns to Matlab. I ran the Onyx Spotlight memory-leak test on
% Showtime.mex and it didn't detect any leaks. So I'm guessing that the
% measured leak is in variables allocated by Matlab through the mx calls
% inside Showtime.mex.
% 
% The code in the first section below, to compute a grating, and display it
% in a window using Screen 'PutImage', works perfectly at all pixelSizes.
% At present, the Showtime.mex conversion of the same Matlab array to a QuickTime movie
% seems to work correctly only if "bits" (ie specified pixelSize for the movie)
% is 8. (16 and 32 might be ok too.) I expect to fix that soon. 
% However, once you've saved the QuickTime movie,
% you can display it at any depth, i.e. you can set your screen's pixelSize to
% anything you want and QuickTime will do an excellent job of rendering the movie
% appropriately.
% 
% Denis Pelli, 11/99

% dgp 11/7/99 Wrote it, based on Beau Watson's Mathematica qtWrite tutorial.
% dgp 11/9/99 Put movie in root directory.
% dgp 11/13/99 Put movie on desktop. Restrict to 8, 16, or 32 bit pixelSize.
% dgp 11/15/99 Add Showtime 'OpenWindow' and 'CloseWindow'.
% dgp 12/5/99 Synched with ShowtimeDemo.m.
% 4/24/02 awi Exit on PC with message.

if strcmp('PCWIN',computer)
    error('Win: Showtime not yet supported.');
end


clear all
mem=1; % 1 to enable memory diagnostics to check for memory leak
% As of 11/13/99 we're losing about 750 bytes of TempFree memory each time
% ShowtimeDemo.m runs.

if mem
	fprintf('%8ld free, %8ld temp free. After initial CLEAR ALL.\n',bytes,bytes('TempFree'));
end

% window
whichScreen=0;
window=Showtime('OpenWindow',whichScreen);
%window=Screen('GetMatlabWindow');
white=GrayIndex(window,1);
black=GrayIndex(window,0);
gray=GrayIndex(window,0.5);
if round(gray)==white
	gray=black;
end
inc=white-gray;
Screen(window,'FillRect',gray);

% grating
frames=90;
[x,y]=meshgrid(-50:50,-50:50);
angle=17*pi/180; % arbitrary 17 deg orientation
f=0.03*2*pi;
a=cos(angle)*f;
b=sin(angle)*f;
rate=FrameRate(window);
hz=2;
grating=uint8(zeros(size(x))); % uint8 takes up 1/8 the space as double
grating(:,:,frames)=grating;
for i=1:frames
 	m=exp(-((x/25).^2)-((y/25).^2)).*sin(a*x+b*y+2*pi*(i/rate)*hz);
	t=8*(i-(frames+1)/2)/frames;
	grating(:,:,i)=gray+inc*m*exp(-t^2);
	Screen(window,'PutImage',grating(:,:,i));
end
if mem
%	fprintf('%8ld free, %8ld temp free. After creating array.\n',bytes,bytes('TempFree'));
end

% Save the QuickTime movie file in the user's desktop.
s=matlabroot;
n=findstr(filesep,s);
root=s(1:n(1));
cd(fullfile(root,'Desktop Folder',''));
filename='DriftingGrating.mov';

% clear grating
% grating=[128 0;0 128];

% Make the movie, and save it to disk as a QuickTime movie file.
rows=size(grating,1);
cols=size(grating,2);
trackTimescale=600;
frameDuration=round(trackTimescale/FrameRate(whichScreen)); 
preload=1;
Screen(window,'FillRect',white);
movie=Showtime('MovieCreate',filename,window);
Showtime('VideoTrackCreate',movie,rows,cols,trackTimescale,preload);
Showtime('VideoMediaCreate',movie);
bits=Screen(window,'PixelSize');
if bits<8;
	bits=8;
end
if mem
%	fprintf('%8ld free, %8ld temp free. Before adding frames.\n',bytes,bytes('TempFree'));
end
Showtime('VideoMediaSamplesAdd',movie,grating,bits,frameDuration);
if mem
%	fprintf('%8ld free, %8ld temp free. After adding frames.\n',bytes,bytes('TempFree'));
end
Showtime('VideoMediaSave',movie);
Showtime('VideoTrackSamplesSet',movie);
Showtime('VideoTrackSave',movie);
%Showtime('MovieUserDataAdd',movie,'A movie demo.','inf');
if mem
%	fprintf('%8ld free, %8ld temp free. Before saving movie.\n',bytes,bytes('TempFree'));
end
Showtime('MovieSave',movie);
fprintf('Your new %d-bit QuickTime movie "%s" is on your desktop.\n',bits,filename);
if mem
%	fprintf('%8ld free, %8ld temp free. After saving movie.\n',bytes,bytes('TempFree'));
end
Showtime('CloseWindow',window);
if mem
	fprintf('%8ld free, %8ld temp free. After closing window.\n',bytes,bytes('TempFree'));
end

% show the movie
movie=Showtime('GetFromFile',filename);
desc=Showtime('GetImageDescription',movie) % report various useful facts about our movie
Showtime('LoadIntoRam',movie);
window=Showtime('OpenWindow',whichScreen);
Showtime('Window',movie,window);
%Screen(window,'FillRect',gray);
%Showtime('Scale',movie,2);
for i=1:1;
	Showtime('Show',movie);
end
Showtime('Dispose',movie);
Showtime('CloseWindow',window);
if mem
	fprintf('%8ld free, %8ld temp free. Done.\n',bytes,bytes('TempFree'));
end
if mem
	clear all
	fprintf('%8ld free, %8ld temp free. After CLEAR ALL.\n',bytes,bytes('TempFree'));
end
