function[AwaFileMat,PwaFileMat]=WaveApp(action,FileMat)

%   WaveApp
%
%   Das Programm erzeut 2 files (Endungen: amplitude wavelet = awa; phaselock wavelet = pwa)
%   in denen die spektrale Amplitude (awa) und die spektrale Phaselock (pwa)
%   abgespeichert werden (vgl Tallon et al, 1997)
%   (die Matrix awa kann mit der Pruefgroesse R auf Signifikanz
%   ueberpruft werden (vgl. Dissertation Wienbruch, Kapitel 2, Formel 51)

%   This software is protected by german copyright and international treaties.             
%   (c) Gruber & Keil, 1999; Junghoefer 2000 All Rights Reserved.                     
%                                                                                          
%   THIS SOFTWARE AND DOCUMENTATION IS PROVIDED "AS IS," AND COPYRIGHT HOLDERS MAKE        
%   NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO,    
%   WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY PARTICULAR PURPOSE OR THAT THE USE    
%   OF THE SOFTWARE OR DOCUMENTATION WILL NOT INFRINGE ANY THIRD PARTY PATENTS, COPYRIGHTS,
%   TRADEMARKS OR OTHER RIGHTS. COPYRIGHT HOLDERS WILL NOT BE LIABLE FOR ANY DIRECT,       
%   INDIRECT, SPECIAL OR CONSEQUENTIAL DAMAGES ARISING OUT OF ANY USE OF THE SOFTWARE      
%   OR DOCUMENTATION.                                

if nargin<2; FileMat=[]; end
if nargin<1; action='InitializeMenu'; end

%================= global variables =============================================
global hWaveMenuFig

if strcmp(action,'InitializeMenu'),
	
	%============================================================================
	%=========== INITIALIZE MENU ================================================
	%============================================================================
	
	%============== figure with graphs =========================		
	if strcmp(computer,'MAC2')	
       	MenuFigPosVec=[.8 0 .2 1];
	elseif strcmp(computer,'PCWIN') 
       	MenuFigPosVec=[.78 .05 .2 .82];
	elseif strcmp(computer,'SOL2')| strcmp(computer,'GLNX86')
       	MenuFigPosVec=[.8 .01 .2 .91];		
	end

	%============== figure with menu ===========================		
	WaveMenuFigString=['Wavelet Menu:'];
   	hWaveMenuFig=figure('Units','normal','Position',MenuFigPosVec,'Name',WaveMenuFigString,'Color',[.99 .99 .99],'NumberTitle','off');
	
	% Information for all buttons (and menus)
	labelColor=[0.8 0.8 0.8];
	yInitPos=0.90;
	menutop=0.955;
	btnTop = 0.6;
	top=0.75;
	left=0.125;
	btnWid=0.8;
	btnHt=0.04;
	btnHt_edit=0.050;
	textHeight = 0.05;			% not used !!!!!!!!!!!
	textWidth = 0.06;
	% Spacing between the button and the next command's label
	spacing=0.015;
	
	%====================================
	% The CONSOLE frame
	frmBorder=0.05; 
	frmLeft=left-frmBorder;
	frmBottom=0.010; 
	frmHeight = 1-3.*frmBottom; 
	frmWidth = btnWid+2*frmBorder;
	yPos=frmBottom-frmBorder;
	frmPos=[frmLeft frmBottom frmWidth frmHeight];
	ConsoleRGBColor=[0.45 0.45 0];
	hConsolFrame=uicontrol( ...
	'Style','frame', ...
	'Units','normalized', ...
	'Position',frmPos, ...
	'BackgroundColor',ConsoleRGBColor);
	
	%==================================================
   	btnNumber=1;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	labelPos=[left yPos-btnHt_edit./2 btnWid btnHt_edit./2];
	h = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',labelPos, ...
	'Horiz','center', ...
	'String','=== Calculate Wavelet Menu ===', ...
	'Interruptible','off', ...
	'Background',ConsoleRGBColor, ...
	'Foreground','black');
	
 	%==============================================================
	% OpenAppFiles (AppSCADS-Session-Format)==============================
	%==============================================================
	btnNumber=2;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''OpenAppFiles'')';
	hOpenAppFilesMenu=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Open App File(s) ', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','on');
	if ~isempty(FileMat)
		set(hOpenAppFilesMenu,'Userdata',FileMat);
	end
	%==============================================================
	% Set Result Path =============================================
	% callback: SetResultPath =====================================
	%==============================================================
	
	btnNumber=2;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left+btnWid./2 yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''SetResultPath'')';
	hSetResultPath=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Result Path', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%======================================================
	% Run Wavelet Analysis ======================================
	% callback: RunWave ========================================
	%======================================================
	btnNumber=3;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''RunWave'')';
	hRunWaveMenu=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Run Wavelet', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','off');
	if ~isempty(FileMat)
		set(hRunWaveMenu,'Enable','on');
	end

	%===============================================
	% SavePhaseStatus (radio button)=============================
	% callback: SavePhaseStatus ================================
	%===============================================
	btnNumber=3;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left+btnWid./2 yPos-btnHt btnWid./2 btnHt];
	
	hSavePhaseStatus=uicontrol( ...
	'Style','radiobutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Save Phaselock', ...
	'Value',1,...
	'Enable','on');
	
	%===============================================
	% SubERPStatus (radio button)===========================
	% callback: SubERPStatus ==============================
	%===============================================
	btnNumber=4;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt btnWid./2 btnHt];
	
	hSubERPStatus=uicontrol( ...
	'Style','radiobutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Subtract EP/EF', ...
	'Value',0,...
	'Enable','on');
	
	
	%===============================================
	% PadDataStatus (radio button)=============================
	% callback: PadDataStatus ================================
	%===============================================
	btnNumber=4;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left+btnWid./2 yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''PadDataStatus'')';
	
	hPadDataStatus=uicontrol( ...
	'Style','radiobutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Data Padding', ...
	'Value',1,...
	'Callback',callbackStr,...
	'Enable','on');
	
	%=================================================
	% SetStartF0 (edit text)==================================
	% callback: SetStartF0 ==================================
	%==================================================
   	btnNumber=6;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos 2.*btnWid./3 btnHt_edit./2];
    
	hSetStartF0Text = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Min. Frequency [Hz]:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetStartF0'')';
    
	hSetStartF0Menu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','10','Value',10, ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%==============================================================
	% SetEndF0 (edit text)==================================
	% callback: SetEndF0 ===================================
	%==============================================================
   	btnNumber=6;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt_edit./2 2.*btnWid./3 btnHt_edit./2];
    
	hSetEndF0Text = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Max. Frequency [Hz]:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos-btnHt_edit./2 btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetEndF0'')';
    
	hSetEndF0Menu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','50','Value',50, ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%==================================================
	% SetDeltaF0 (edit text)==================================
	% callback: SetDeltaF0 ===================================
	%===================================================
   	btnNumber=7;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos 2.*btnWid./3 btnHt_edit./2];
    
	hSetDeltaF0Text = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Delta Frequency [Hz]:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetDeltaF0'')';
    
	hSetDeltaF0Menu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','1','Value',1, ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%================================================
	% SetNTap (edit text)==================================
	% callback: SetNTap ==================================
	%================================================
   	btnNumber=7;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt_edit./2 2.*btnWid./3 btnHt_edit./2];
    
	hSetNTapText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Number of Tap. Points:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos-btnHt_edit./2 btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetNTap'')';
    
	hSetNTapMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','10','Value',10, ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%==================================================
	% SetSigmaM (edit text)==================================
	% callback: SetSigmaM ===================================
	%===================================================
   	btnNumber=8;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos 2.*btnWid./3 btnHt_edit./2];
    
	hSetSigmaMText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','M (Sigma Frequ.):',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetSigmaM'')';
    
	hSetSigmaMMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','7','Value',7, ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%=================================================
	% SetNStore (edit text)==================================
	% callback: SetNStore ===================================
	%==================================================
   	btnNumber=8;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt_edit./2 2.*btnWid./3 btnHt_edit./2];
    
	hSetNStoreText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Storage Interval:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos-btnHt_edit./2 btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetNStore'')';
    
	hSetNStoreMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','1','Value',1, ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%=================================================
	% SetNChan (edit text)==================================
	% callback: SetNChan ===================================
	%==================================================
   	btnNumber=9;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos 2.*btnWid./3 btnHt_edit./2];
    
	hSetNChanText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Number of Sensors:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetNChan")';
    
	hSetNChanMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','0','Value',0, ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=================================================
	% SetNChanExtra (edit text)=============================
	% callback: SetNChan =================================
	%==================================================
   	btnNumber=9;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt_edit./2 2.*btnWid./3 btnHt_edit./2];
    
	hSetNChanExtraText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Number of Ext. Sensors:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos-btnHt_edit./2 btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetNChanExtra")';
    
	hSetNChanExtraMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','0','Value',0, ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=================================================
	% SetNPoints (edit text)==================================
	% callback: SetNPoints ===================================
	%==================================================
   	btnNumber=10;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos 2.*btnWid./3 btnHt_edit./2];
    
	hSetNPointsText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Number of Orig. Points:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetNPoints'')';
    
	hSetNPointsMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','0','Value',0, ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=================================================
	% SetNPointsPad (edit text)==================================
	% callback: SetNPointsPad ================================
	%==================================================
   	btnNumber=10;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt_edit./2 2.*btnWid./3 btnHt_edit./2];
    
	hSetNPointsPadText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Number of Pad. Points:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos-btnHt_edit./2 btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetNPointsPad'')';
    
	hSetNPointsPadMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','0','Value',0, ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=================================================
	% SetNTrials (edit text)==================================
	% callback: SetNTrials ===================================
	%==================================================
   	btnNumber=11;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos 2.*btnWid./3 btnHt_edit./2];
    
	hSetNTrialsText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Number of Trials:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetNTrials'')';
    
	hSetNTrialsMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','0','Value',0, ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=================================================
	% SetSampRate (edit text)================================
	% callback: SetSampRate ================================
	%==================================================
   	btnNumber=11;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt_edit./2 2.*btnWid./3 btnHt_edit./2];
    
	hSetSampRateText = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','left', ...
	'Foreground','black', ...
	'String','Sampling Rate [Hz]:',...
	'Enable','on');
    
	btnPos=[left+2.*btnWid./3 yPos-btnHt_edit./2 btnWid./3 btnHt_edit./2];
	callbackStr = 'WaveApp(''SetSampRate'')';
    
	hSetSampRateMenu = uicontrol( ...
	'Style','edit', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'Horiz','center', ...
	'Background','white', ...
	'Foreground','black', ...
	'String','0','Value',0, ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=======================================================
	% SurfWaveMatFrequ =======================================
	% callback: SurfWaveMatFrequ =================================
	%=======================================================
	
	btnNumber=12;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt btnWid./2 btnHt./2];
	callbackStr = 'WaveApp(''SurfWaveMatFrequ'')';
	hSurfWaveMatFrequ=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Surf Wavelet Frequ.', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=======================================================
	% SurfWaveMatTime =======================================
	% callback: SurfWaveMatTime =================================
	%=======================================================
	
	btnNumber=12;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt./2 btnWid./2 btnHt./2];
	callbackStr = 'WaveApp(''SurfWaveMatTime'')';
	hSurfWaveMatTime=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Surf Wavelet Time', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=======================================================
	% SurfTapMat ===========================================
	% callback: SurfTapMat =====================================
	%=======================================================
	
	btnNumber=12;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left+btnWid./2 yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''SurfTapMat'')';
	hSurfTapMat=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Surf Taper Matrix', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=======================================================
	% SurfTrialStatus ===========================================
	% callback: SurfTrialStatus =====================================
	%=======================================================
	
	btnNumber=13;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt btnWid./2 btnHt];
	hSurfTrialStatus=uicontrol( ...
	'Style','radiobutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Surf Trial', ...
	'Value',0, ...
	'Enable','on');
	
	%=======================================================
	% SurfERPStatus ===========================================
	% callback: SurfERPStatus =====================================
	%=======================================================
	
	btnNumber=13;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left+btnWid./2 yPos-btnHt btnWid./2 btnHt];
	hSurfERPStatus=uicontrol( ...
	'Style','radiobutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Surf ERP', ...
	'Value',0, ...
	'Enable','on');
	
	%=======================================================
	% SurfERPStatus ===========================================
	% callback: SetEegMegStatus =================================
	%=======================================================
	
	btnNumber=14;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''SetEegMegStatus'')';
	hSetEegMegStatus=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','EEG', ...
	'Value',1, ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%=======================================================
	%=======================================================
	%=======================================================
   	btnNumber=15;     
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	labelPos=[left yPos-btnHt_edit./2 btnWid btnHt_edit./2];
	h = uicontrol( ...
	'Style','text', ...
	'Units','normalized', ...
	'Position',labelPos, ...
	'Horiz','center', ...
	'String','=== Avg. & Diff. Wavelet Menu ===', ...
	'Interruptible','off', ...
	'Background',ConsoleRGBColor, ...
	'Foreground','black');
	
	%==============================================================
	% OpenWaFiles (Wa-Format)==========================================
	%==============================================================
	
	btnNumber=16;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''OpenWaFiles'')';
	hOpenWaFilesMenu=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Open Wa File(s) ', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%==============================================================
	% hSetAwaPwaStatus================================================
	%==============================================================
	btnNumber=16;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left+btnWid./2 yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''SetAwaPwaStatus'')';
	hSetAwaPwaStatus=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Amplitude', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','on');
	
	%==============================================================
    % AverageWaFiles (Wa-Format)===================================
	%==============================================================
	btnNumber=17;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''AvgWaFiles'')';
	hAvgWaFilesMenu=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Avg. Wa Files', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%==============================================================
    %	DifferenceWaFiles (Wa-Format)==============================
	%==============================================================
	btnNumber=17;
	yPos=menutop-(btnNumber-1)*(btnHt+spacing);
	btnPos=[left+btnWid./2 yPos-btnHt btnWid./2 btnHt];
	callbackStr = 'WaveApp(''DiffWaFiles'')';
	hDiffWaFilesMenu=uicontrol( ...
	'Style','pushbutton', ...
	'Units','normalized', ...
	'Position',btnPos, ...
	'String','Diff. Wa Files', ...
	'Userdata',[], ...
	'Callback',callbackStr,...
	'Enable','off');
	
	%=======================================================
	%=======================================================
	hAwaMat=uicontrol('Visible','off');
	hPwaMat=uicontrol('Visible','off');
	hWaveMat=uicontrol('Visible','off');
	hF0FrequVec=uicontrol('Visible','off');
	hF0IndexVec=uicontrol('Visible','off');
	%=======================================================
	%=======================================================
	
	NonUse=0;
	hList=[hOpenAppFilesMenu hSetResultPath hRunWaveMenu,hAwaMat,hPwaMat ...					%5
	hSetNTapText hSetNTapMenu hSetStartF0Text hSetStartF0Menu hSetEndF0Text ...					%10
	hSetEndF0Menu hSetDeltaF0Text hSetDeltaF0Menu hSetSigmaMText hSetSigmaMMenu ...				%15
	hSavePhaseStatus hSetNStoreText hSetNStoreMenu hSubERPStatus hPadDataStatus ...				%20
	hOpenWaFilesMenu hSetAwaPwaStatus hAvgWaFilesMenu hDiffWaFilesMenu hWaveMat ...				%25
	hSetNChanText hSetNChanMenu hSetNPointsText hSetNPointsMenu hSetNTrialsText ...				%30
	hSetNTrialsMenu hSetSampRateText hSetSampRateMenu hSetNPointsPadText hSetNPointsPadMenu ...	%35
	NonUse NonUse hSurfTapMat hSurfTrialStatus hSurfERPStatus ...								%40
	hF0FrequVec hF0IndexVec hSurfWaveMatFrequ hSurfWaveMatTime NonUse ...						%45
	hSetNChanExtraText hSetNChanExtraMenu hSetEegMegStatus];									%50												
	
	set(hWaveMenuFig,'UserData',hList);
	
	%============================================================================	
	%============ set EEG / MEG Status
	%============================================================================
	
elseif strcmp(action,'SetEegMegStatus'),

	hList=get(hWaveMenuFig,'Userdata');
	hSetEegMegStatus=hList(48);
	EegMegStatusString=get(hSetEegMegStatus,'String');
	if strcmp(EegMegStatusString,'EEG')
		EegMegStatusString='MEG';
		EegMegStatusVal=2;
	else
		EegMegStatusString='EEG';
		EegMegStatusVal=1;
	end
	set(hSetEegMegStatus,'String',EegMegStatusString,'Value',EegMegStatusVal);
	
elseif  strcmp(action,'AppFileMatrixInfo') | strcmp(action,'WaFileMatrixInfo') 

	hList=get(hWaveMenuFig,'Userdata'); 
	if  strcmp(action,'AppFileMatrixInfo')
		hListNo=1;
	elseif strcmp(action,'WaFileMatrixInfo') 
		hListNo=21;
	end
	FileMat=get(hList(hListNo),'Userdata')
	NFiles=size(FileMat,1);
	
	if NFiles>1
		fprintf(1,'\n'); clc;
		Message=char(['The actual file matrix contains the following ',int2str(NFiles) ' files:']);
		fprintf(1,'The actual file matrix contains the following %g files:\n\n',NFiles);
	else
		return;
	end
	Message=char(Message,' ');
	Message=char(Message,FileMat);
	Message=char(Message,'');
	Message=char(Message,'');
	hmsgbox=msgbox(Message,'Info','help'); 
	disp(FileMat)
	
elseif strcmp(action,'PadDataStatus'),

	hList=get(hWaveMenuFig,'Userdata');
	PadDataStatus=get(hList(20),'Value');
	NPoints=get(hList(29),'Value');
	if NPoints==0; return; end
	if PadDataStatus
		NPointsPad=2;
		Pot = 2;
		while NPointsPad<NPoints;
			Pot=Pot+1;
			NPointsPad=2^Pot;
		end
	else
		NPointsPad=NPoints;
	end
	set(hList(35),'Value',NPointsPad,'String',sprintf('%g',NPointsPad));
	WaveApp('SetStartF0')
	WaveApp('SetEndF0')
	WaveApp('SetDeltaF0')
	
elseif strcmp(action,'OpenAppFiles') | strcmp(action,'SetAppFiles'),

	hList=get(hWaveMenuFig,'Userdata');
	ResultPath=get(hList(2),'Userdata');
	NTap=get(hList(7),'Value');
	StartF0=get(hList(9),'Value');
	EndF0=get(hList(11),'Value');
	DeltaF0=get(hList(13),'Value');
	PadDataStatus=get(hList(20),'Value');
	if strcmp(action,'OpenAppFiles')
		FileExt='*.app*'; 
		String='Choose an App. file (*.app*):';
		DefFilePath=SetDefPath(1,FileExt)
		[NFiles,FileMat,VersionString,BatchFilePath]=ReadFileNames(FileMat,DefFilePath,String);
		if NFiles==0; return; end
		set(hList(1),'Userdata',FileMat);
		set(hList(3),'Enable','on');	%RunWave
		set(hList(43),'Enable','on');	%SurfWaveMatFrequ
		set(hList(44),'Enable','on');	%SurfWaveMatTime
	elseif strcmp(action,'SetAppFiles')
		FileMat=get(hList(1),'Userdata');
	end
	WaveApp('SetResultFilePath')
	WaveApp('AppFileMatrixInfo')
	[File,Path,FilePath]=GetFileNameOfMat(FileMat,1);
	[DataMat,Version,LHeader,ScaleBins,NChan,NPoints,NTrials,SampRate,chAvgRef,File,Path,FilePath,EegMegStatus,NChanExtra,AppFileFormatVal]=ReadAppData(FilePath,0);
	set(hList(27),'Value',NChan,'String',sprintf('%g',NChan));
	set(hList(47),'Value',NChanExtra,'String',sprintf('%g',NChanExtra));
	set(hList(29),'Value',NPoints,'String',sprintf('%g',NPoints));
	set(hList(31),'Value',NTrials,'String',sprintf('%g',NTrials));
	set(hList(33),'Value',SampRate,'String',sprintf('%g',SampRate));
	EegMegStatusString=get(hList(48),'String');
	if (EegMegStatus==2 & strcmp(EegMegStatusString,'EEG')) | (EegMegStatus==1 & strcmp(EegMegStatusString,'MEG'))
		WaveApp('SetEegMegStatus')
		EegMegStatusString=get(hList(48),'String');
	end
	WaveApp('PadDataStatus')
	WaveApp('SetNTap')
	NTap=get(hList(7),'Value');
	if NTap>0
		set(hList(38),'Enable','on');	%SurfNTapMat
	end

elseif strcmp(action,'OpenWaFiles'),

	hList=get(hWaveMenuFig,'Userdata');
	ResultPath=get(hList(2),'Userdata');
	hSetAwaPwaStatus=hList(22);
	AwaPwaStatus=get(hSetAwaPwaStatus,'String');
	if strcmp(AwaPwaStatus,'Amplitude')
		FileExt='*.awa*'; 
		String='Choose an amplitude file (*.awa*):';
	elseif strcmp(AwaPwaStatus,'Phaselock')
		FileExt='*.pwa*'; 
		String='Choose a phaselock file (*.pwa*):';
	end
	DefFilePath=SetDefPath(1,FileExt);
	[NFiles,FileMat,VersionString,BatchFilePath]=ReadFileNames(FileMat,DefFilePath,String);
	if NFiles==0; return; end
	set(hList(21),'Userdata',FileMat);
	WaveApp('WaFileMatrixInfo')
	if NFiles>1
		set(hList(23),'Enable','on');	%Avg
		set(hList(24),'Enable','on');	%Diff
	else
		set(hList(23),'Enable','off');
		set(hList(24),'Enable','off');
	end

elseif strcmp(action,'SetAwaPwaStatus'),

	hList=get(hWaveMenuFig,'Userdata');
	hSetAwaPwaStatus=hList(22);
	AwaPwaStatus=get(hSetAwaPwaStatus,'String');
	if strcmp(AwaPwaStatus,'Amplitude')
		AwaPwaStatus='Phaselock';
		set(hList(21),'String','Open pwa files:'); 
	elseif strcmp(AwaPwaStatus,'Phaselock')
		AwaPwaStatus='Amplitude';
		set(hList(21),'String','Open awa files:'); 
	end
	set(hSetAwaPwaStatus,'String',AwaPwaStatus);
	set(hList(21),'Userdata',[]);
	set(hList(23),'Enable','off');	%Avg
	set(hList(24),'Enable','off');	%Diff
	
elseif strcmp(action,'AvgWaFiles')

	hList=get(hWaveMenuFig,'Userdata');
	ResultPath=get(hList(2),'Userdata');
	FileMat=get(hList(21),'Userdata');
	hSetAwaPwaStatus=hList(22);
	AwaPwaStatus=get(hSetAwaPwaStatus,'String');
	[SaveFile,SavePath,SaveFilePath]=GetFileNameOfMat(FileMat,1);
	if strcmp(AwaPwaStatus,'Amplitude')
		initFile='GM.awa';
	elseif strcmp(AwaPwaStatus,'Phaselock')
		initFile='GM.pwa';
	end
	[SaveFile,SavePath,SaveFilePath]=WriteFilePath(initFile,'Choose avg. file name:');
	if ~isempty(ResultPath)
		SaveFilePath=[ResultPath,SaveFile];
	end
	[DataMat,SaveFilePath,SizeDataMat,NDim,Version,SampRate,AvgRef,StartF0,EndF0,DeltaF0,NTap,SigmaM, ...
    EegMegStatus,NChanExtra,AmpPhaseStatus,NTrials]=AvgWaveletData(FileMat,SaveFilePath,2);
	
elseif strcmp(action,'DiffWaFiles')

	hList=get(hWaveMenuFig,'Userdata');
	ResultPath=get(hList(2),'Userdata');
	FileMat=get(hList(21),'Userdata');
	hSetAwaPwaStatus=hList(22);
	AwaPwaStatus=get(hSetAwaPwaStatus,'String');
	[SaveFile,SavePath,SaveFilePath]=GetFileNameOfMat(FileMat,1);
	if strcmp(AwaPwaStatus,'Amplitude')
		initFile='Diff.awa';
	elseif strcmp(AwaPwaStatus,'Phaselock')
		initFile='Diff.pwa';
	end
    if size(FileMat,1)==2
	    [SaveFile,SavePath,SaveFilePath]=WriteFilePath(initFile,'Choose diff. file name:');
    else
        SaveFilePath=[];
    end
	if ~isempty(ResultPath)
		SaveFilePath=[ResultPath,SaveFile];
	end
	[DataMat,SaveFilePath,SizeDataMat,NDim,Version,SampRate,AvgRef,StartF0,EndF0,DeltaF0,NTap,SigmaM, ...
    EegMegStatus,NChanExtra,AmpPhaseStatus,NTrials]=DiffWaveletData(FileMat,SaveFilePath);
	
elseif strcmp(action,'SetResultPath'),

	hList=get(hWaveMenuFig,'Userdata');
	FileMat=get(hList(1),'Userdata');
	ResultPath=[];
	WriteDlgTitle = 'Save Path';
	WriteDef = {pwd};
	WritePrompt = {'Enter full path to save data:'};
	while isempty(ResultPath)
		WritePath=inputdlg(WritePrompt,WriteDlgTitle,1,WriteDef);
		if isempty(WritePath); return; end
		WritePath=char(WritePath);
		if ~strcmp(WritePath(length(WritePath)),filesep)
			WritePath=[WritePath,filesep];
		end
		if exist(WritePath)
			ResultPath=WritePath;
		end
	end
	disp(ResultPath)
	set(hList(2),'Userdata',ResultPath);
	if ~isempty(FileMat)
		WaveApp('SetResultFilePath')
	end

elseif strcmp(action,'SetResultFilePath'),

	hList=get(hWaveMenuFig,'Userdata');
	FileMat=get(hList(1),'Userdata');
	NFiles=size(FileMat,1);
	ResultPath=get(hList(2),'Userdata');
	AwaFileMat=[];
	PwaFileMat=[];
	for FileInd=1:NFiles
		ExtStr=[];
		[File,Path,FilePath]=GetFileNameOfMat(FileMat,FileInd);
		Tmp=findstr(File,'.app');
		if isempty(Tmp)
			Tmp=length(File)+1;
		else
			if Tmp<length(File+5)
				ExtStr=[File(Tmp+4:length(File))];
			end
		end
		if ~isempty(ResultPath); Path=ResultPath; end
		[AwaFileMat]=AddFileName2Mat(AwaFileMat,[Path,File(1:Tmp-1),'.awa',ExtStr],0);
		[PwaFileMat]=AddFileName2Mat(PwaFileMat,[Path,File(1:Tmp-1),'.pwa',ExtStr],0);
	end
	fprintf('Save Amplitude Matrix to file:\n\n')
	disp(AwaFileMat)
	fprintf('\n\n\n')
	fprintf('Save Phaselock Matrix to file:\n\n')
	disp(PwaFileMat)
	set(hList(4),'Userdata',AwaFileMat);
	set(hList(5),'Userdata',PwaFileMat);
	
elseif strcmp(action,'SetStartF0') | ...
	strcmp(action,'SetEndF0') | ...
	strcmp(action,'SetDeltaF0'),	
	
	hList=get(hWaveMenuFig,'Userdata');
	SampRate=get(hList(33),'Value');
	NPointsPad=get(hList(35),'Value');
	FFac=SampRate./NPointsPad;
	StartF0=get(hList(9),'Value');
	StartF0Str=get(hList(9),'String');
	EndF0=get(hList(11),'Value');
	EndF0Str=get(hList(11),'String');
	DeltaF0=get(hList(13),'Value');
	DeltaF0Str=get(hList(13),'String');
	
	if strcmp(action,'SetStartF0')
		[StartF0,StartF0Str]=String2Value(StartF0Str,StartF0,0,EndF0,[],1);
		StartF0=FFac.*round(StartF0./FFac);
		set(hList(9),'Value',StartF0,'String',sprintf('%g',round(StartF0.*10^4)./10^4));
	elseif strcmp(action,'SetEndF0')
		[EndF0,EndF0Str]=String2Value(EndF0Str,EndF0,StartF0,SampRate./2,[],1);
		EndF0=FFac.*round(EndF0./FFac);
		set(hList(11),'Value',EndF0,'String',sprintf('%g',round(EndF0.*10^4)./10^4));
	elseif strcmp(action,'SetDeltaF0')
		[DeltaF0,DeltaF0Str]=String2Value(DeltaF0Str,DeltaF0,FFac,EndF0-StartF0,0,1);
		DeltaF0=FFac.*round(DeltaF0./FFac);
		set(hList(13),'Value',DeltaF0,'String',sprintf('%g',round(DeltaF0.*10^4)./10^4));
	end
	F0FrequVec=[StartF0:DeltaF0:EndF0];
	NF0Vec=length(F0FrequVec);
	StartF0Index=StartF0./FFac+1;
	EndF0Index=EndF0./FFac+1;
	F0IndexVec=[StartF0Index:1:EndF0Index];
	set(hList(41),'Value',NF0Vec,'Userdata',F0FrequVec);
	set(hList(42),'Value',NF0Vec,'Userdata',F0IndexVec);
	
elseif strcmp(action,'SetNTap'),	

	hList=get(hWaveMenuFig,'Userdata');
	NTap=get(hList(7),'Value');
	NTapStr=get(hList(7),'String');
	NPoints=get(hList(29),'Value');
	if NPoints==0
		[NTap,NTapStr]=String2Value(NTapStr,NTap,[],[],1);
	else
		[NTap,NTapStr]=String2Value(NTapStr,NTap,0,floor(NPoints./2),1,1);
	end
	set(hList(7),'Value',NTap,'String',NTapStr);
	if NTap==0;
		TapMat=[];
		set(hList(38),'Userdata',TapMat,'Enable','off');
	else
		if NPoints>0
			set(hList(38),'Enable','on');
		end
	end

elseif strcmp(action,'SetNStore'),	

	hList=get(hWaveMenuFig,'Userdata');
	NStore=get(hList(18),'Value');
	NStoreStr=get(hList(18),'String');
	[NStore,NStoreStr]=String2Value(NStoreStr,NStore,1,[],1);
	set(hList(18),'Value',NStore,'String',NStoreStr);
	
elseif strcmp(action,'SetSigmaM'),	

	hList=get(hWaveMenuFig,'Userdata');
	SigmaM=get(hList(15),'Value');
	SigmaMStr=get(hList(15),'String');
	[SigmaM,SigmaMStr]=String2Value(SigmaMStr,SigmaM,1,10,1,1);
	set(hList(15),'Value',SigmaM,'String',SigmaMStr);
	
elseif strcmp(action,'CalcWaveMat'),	

	hList=get(hWaveMenuFig,'Userdata');
	NF0Vec=get(hList(42),'Value');
	F0IndexVec=get(hList(42),'Userdata');
	SigmaM=get(hList(15),'Value');
	NPointsPad=get(hList(35),'Value');
	WaveMat=zeros(NPointsPad,NF0Vec);
	for F0Ind=1:NF0Vec
		SigmaF=F0IndexVec(F0Ind)./SigmaM;
		A=1/(sqrt(SigmaF.*sqrt(pi)));
		for PointInd=1:NPointsPad;
			WaveMat(PointInd,F0Ind)=A.*exp(-((PointInd-F0IndexVec(F0Ind)).^2)./(2.*SigmaF.^2));
		end
	end 
	fprintf(1,'Size of Wavelet Matrix: %g x %g\n',NPointsPad,NF0Vec)
	set(hList(25),'Userdata',WaveMat);
	
elseif strcmp(action,'SurfWaveMatFrequ') | strcmp(action,'SurfWaveMatTime'),	

	hList=get(hWaveMenuFig,'Userdata');
	WaveApp('CalcWaveMat')
	WaveMat=get(hList(25),'Userdata');
	SampRate=get(hList(33),'Value');
	NPointsPad=get(hList(35),'Value');
	NF0Vec=get(hList(41),'Value');
	F0FrequVec=get(hList(41),'Userdata');
	F0FrequVec=round(F0FrequVec.*100)./100;
	if strcmp(action,'SurfWaveMatFrequ')
		FigPos=[.5 .5 .5 .5];
		Name='Wavelet in Frequency Domain';
		MaxPoint=NPointsPad./2+1;
		MaxY=SampRate./2;
		TmpX=char('Frequency [Hz]:');
		TmpTmpX=char(TmpX,'(Min.-Max. Frequ.)');
		TmpTmpY=char(TmpX,'(0-Nyquist Frequ.)');
	elseif strcmp(action,'SurfWaveMatTime')
		FigPos=[0 .5 .5 .5];
		Name='Wavelet in Time Domain';	
		WaveMat=abs(ifft(WaveMat,NPointsPad,1));
		MaxPoint	=NPointsPad;
		MaxY=NPointsPad.*1000./SampRate;
		TmpX=char('Frequency [Hz]:');
		TmpTmpX=char(TmpX,'(Min.-Max. Frequ.)');
		TmpTmpY=char('Time [ms]:');
	end
	hFig=figure('Units','normal','NumberTitle','off','Position',FigPos,'Name',Name);
	surf(WaveMat(1:MaxPoint,1:NF0Vec))
	axis([1,NF0Vec,1,MaxPoint,min(min(WaveMat)),max(max(WaveMat))])
	XTickVec=get(gca,'xtick');
 	XTickLabelVec=F0FrequVec(XTickVec);
	set(gca,'XTickLabel',XTickLabelVec);
	YTickVec=get(gca,'ytick');
	TmpVec=linspace(0,MaxY,MaxPoint);
	TmpVec=round(TmpVec.*100)./100;
	set(gca,'YTickLabel',TmpVec(YTickVec));
	xlabel(TmpTmpX)
	ylabel(TmpTmpY)
	zlabel('Amplitude [ ]:')
	title(Name)
	
elseif strcmp(action,'CalcTapMat'),	

	hList=get(hWaveMenuFig,'Userdata');
	NTap=get(hList(7),'Value');
	NPoints=get(hList(29),'Value');
	NChan=get(hList(27),'Value');
	NChanExtra=get(hList(47),'Value');
	NChanAct=NChan+NChanExtra;
	TapVec1=(cos(linspace(pi./2,pi,NTap)).^2)';
	TapVec2=(cos(linspace(0,pi./2,NTap)).^2)';
	TapMat=ones(NChanAct,NPoints);
	TapMat(:,1:NTap)=TapMat(:,1:NTap)*diag(TapVec1);
	TapMat(:,NPoints-NTap+1:NPoints)=TapMat(:,NPoints-NTap+1:NPoints)*diag(TapVec2);
	set(hList(38),'Userdata',TapMat);
	
elseif strcmp(action,'SurfTapMat'),	

	hList=get(hWaveMenuFig,'Userdata');
	WaveApp('CalcTapMat')
	TapMat=get(hList(38),'Userdata');
	SampRate=get(hList(33),'Value');
	NPoints=get(hList(29),'Value');
	NChanAct=get(hList(27),'Value');
	hFig=figure('Units','normal','NumberTitle','off','Position',[0 .5 .5 .5],'Name','Taper Matrix');
	surf(TapMat)
	axis([1,NPoints,1,NChanAct,min(min(TapMat)),max(max(TapMat))])
	XTickVec=get(gca,'xtick');
	set(gca,'XTickLabel',XTickVec.*1000./SampRate);
	ylabel('Sensor #:')
	xlabel('Time [ms]:')
	zlabel('Amplitude [  ]:')
	
elseif strcmp(action,'RunWave'),	

	hList=get(hWaveMenuFig,'Userdata');
	FileMat=get(hList(1),'Userdata');
	NFiles=size(FileMat,1);
	AwaFileMat=get(hList(4),'Userdata');
	PwaFileMat=get(hList(5),'Userdata');
	
	NTap=get(hList(7),'Value');
	StartF0=get(hList(9),'Value');
	EndF0=get(hList(11),'Value');
	DeltaF0=get(hList(13),'Value');
	NF0Vec=get(hList(41),'Value');
	F0FrequVec=get(hList(41),'Userdata')
	SigmaM=get(hList(15),'Value');
	NPoints=get(hList(29),'Value');
	NPointsPad=get(hList(35),'Value');
	SavePhaseStatus=get(hList(16),'Value');
	NStore=get(hList(18),'Value');
	SubERPStatus=get(hList(19),'Value');
	PadDataStatus=get(hList(20),'Value');
	SurfTrialStatus=get(hList(39),'Value');
	SurfERPStatus=get(hList(40),'Value');
	EegMegStatusString=get(hList(48),'String');
	
	SensorVec=[];
	
	[FileSizeVec]=GetFileSize(FileMat);
	
	Message=char(zeros(9,1));
	hProgressMsgbox=msgbox(Message,'Progress Info:','help')
	hChildProgressMsgbox=get(hProgressMsgbox,'children');
    hChild2ProgressMsgbox=get(hChildProgressMsgbox(2),'children');
    set(hChildProgressMsgbox(length(hChildProgressMsgbox)),'Visible','off');
	for FileInd=1:NFiles
		[File,Path,FilePath]=GetFileNameOfMat(FileMat,FileInd);
		FileMessage=char('Calculate wavelet ...');
		FileMessage=char(FileMessage,'');
		FileMessage=char(FileMessage,['File ',int2str(FileInd),' of ',int2str(NFiles)]);
		FileMessage=char(FileMessage,File);
		FileMessage=char(FileMessage,'');
		if SubERPStatus
			if strcmp(EegMegStatusString,'EEG')
				Message=char(FileMessage,'Calculate evoked potential...');
			else
				Message=char(FileMessage,'Calculate evoked field...');
			end
		else
			Message=FileMessage;
		end
		Message=char(FileMessage,'');
		if strcmp(computer,'MAC2')
			set(hChildProgressMsgbox(2),'String',char(FileMessage,''));
		else
			set(hChild2ProgressMsgbox,'String',char(FileMessage,''));
		end
		pause(.001);
		DataMat=[];	
		if  FileSizeVec(FileInd)<10^6
			[DataMat,Version,LHeader,ScaleBins,NChan,NPoints,NTrials,SampRate,chAvgRef,...
			NonUse,NonUse,NonUse,EegMegStatus,NChanExtra,AppFileFormatVal]=ReadAppData(FilePath,'all');
			NChanAct=NChan+NChanExtra;
			set(hList(27),'String',sprintf('%g',NChan));
			set(hList(29),'String',sprintf('%g',NPoints));
			set(hList(31),'String',sprintf('%g',NTrials));
			set(hList(47),'String',sprintf('%g',NChanExtra));
			if isempty(SensorVec)
				SensorVec=[1:NChanAct];  %Sensors to analyze
			end
			DataMat=DataMat(:,SensorVec,:);
		end
		if SubERPStatus
			%  compute ERP
			fprintf(1,'Read data from file:\n');
			fprintf(1,FilePath);
			fprintf(1,'\n');
			if  ~isempty(DataMat)
				ERP=squeeze(mean(DataMat,1));
			else
				[ERP,Version,LHeader,ScaleBins,NChan,NPoints,NTrials,SampRate,chAvgRef,...
				NonUse,NonUse,NonUse,EegMegStatus,NChanExtra,AppFileFormatVal]=ReadAppData(FilePath);
				NChanAct=NChan+NChanExtra;
				for TrialInd =2:NTrials
					fprintf(1,'%g of %g\n',TrialInd,NTrials)
     				[TrialMat]=ReadAppData(FilePath,TrialInd);
     				ERP=ERP+TrialMat;
					if TrialInd==1 | round(TrialInd/10)==TrialInd/10
						if strcmp(computer,'MAC2')
							set(hChildProgressMsgbox(2),'String',char(Message,['Trial ',int2str(TrialInd),' of ',int2str(NTrials),' ...']));
						else
							set(hChild2ProgressMsgbox,'String',char(Message,['Trial ',int2str(TrialInd),' of ',int2str(NTrials),' ...']));
						end
						pause(.001)
					end
				end
				if isempty(SensorVec)
					SensorVec=[1:NChanAct];  %Sensors to analyze
				end
     			ERP=ERP(SensorVec,:)./NTrials;
			end
			NChanAct=length(SensorVec);
			if SurfERPStatus
				hFig=figure('Units','normal','NumberTitle','off','Position',[0 .5 .5 .5],'Name','Averaged Event Related Potential');
				MinAmp=min(min(ERP));
				MaxAmp=max(max(ERP));
				fprintf('\n\n');
				fprintf('Minimum Amplitude: %g �V\n\n',MinAmp);
				fprintf('Maximum Amplitude: %g �V\n\n',MaxAmp);
				surf(ERP)
				axis([1,NPoints,1,NChanAct,MinAmp,MaxAmp])
				XTickVec=get(gca,'xtick');
				set(gca,'XTickLabel',XTickVec.*1000./SampRate);
				ylabel('Sensor #:')
				xlabel('Time [ms]:')
				zlabel('Amplitude [�V]:')
				title(['Min. Amp: ',sprintf('%g',MinAmp), ' �V;  Max. Amp.: ',sprintf('%g',MaxAmp),' �V'])
				pause(.01)	
			end
		else	%if SubERPStatus
			[NonUse,Version,LHeader,ScaleBins,NChan,NPoints,NTrials,SampRate,chAvgRef,...
			NonUse,NonUse,NonUse,EegMegStatus,NChanExtra,AppFileFormatVal]=ReadAppData(FilePath,0);
			NChanAct=NChan+NChanExtra;
			if isempty(SensorVec)
				SensorVec=[1:NChanAct];  %Sensors to analyze
			end
			if strcmp(computer,'MAC2')
				set(hChildProgressMsgbox(2),'String',Message);
			else
				set(hChild2ProgressMsgbox,'String',Message);			
			end
			pause(.001);
		end	%if SubERPStatus
		fprintf(1,'\n');
  		fprintf(1,'Number of sensors: %g \n',NChanAct);
    	fprintf(1,'Number of points: %g\n\n',NPoints);		
		PadMat=zeros(NChanAct,NPointsPad);
		if NTap>0
			WaveApp('CalcTapMat')
			TapMat=get(hList(38),'Userdata');
		end
		%Calculate Wavelet
		Message=char(FileMessage,'Calculate wavelet ...');
		if strcmp(computer,'SOL2') | strcmp(computer,'PCWIN')
			set(hChild2ProgressMsgbox,'String',Message);
		else
			set(hChildProgressMsgbox(2),'String',Message);
		end
		pause(.001);
		WaveApp('CalcWaveMat')
		WaveMat=get(hList(25),'Userdata');
		
		SumPower = [];
		AvgWaPower = [];
		NF0OnesVec=ones(NF0Vec,1);
		AwaMat=zeros(NChanAct,NPoints,NF0Vec);
		RayMat=AwaMat;
		for TrialInd=1:NTrials
			if TrialInd==1 | floor(TrialInd./10)==TrialInd./10
				fprintf(1,'\n')
				fprintf(1,'Trial %g of %g\n',TrialInd,NTrials);
			else
				fprintf(1,'.')
			end
			Message=char(FileMessage,['Trial ',int2str(TrialInd),' of ',int2str(NTrials)]);
			if strcmp(computer,'SOL2') | strcmp(computer,'PCWIN')
				set(hChild2ProgressMsgbox,'String',Message);
			else
				set(hChildProgressMsgbox(2),'String',Message);
			end
			pause(.001);
			if isempty(DataMat);
				[TrialMat]=ReadAppData(FilePath,TrialInd);
				TrialMat=TrialMat(SensorVec,:);
			else
				TrialMat=squeeze(DataMat(TrialInd,:,:));
			end
			if NTap>0
				TrialMat=TrialMat.*TapMat;
			end
			if SubERPStatus
				PadMat(:,1:NPoints)=TrialMat-ERP;	% ERP Subtract
			else
				PadMat(:,1:NPoints)=TrialMat;
			end
			FFTMat=fft(PadMat,NPointsPad,2);
			for ChanInd=1:NChanAct
				TmpMat=(WaveMat.*(NF0OnesVec*FFTMat(ChanInd,:))')';
				TmpMat=(ifft(TmpMat,NPointsPad,2))';
				TmpMat=TmpMat(1:NPoints,:);
				AwaMat(ChanInd,:,:)=abs(TmpMat);
				if SavePhaseStatus == 1
					RayMat(ChanInd,:,:)=TmpMat./squeeze(AwaMat(ChanInd,:,:));
				end
			end
			if TrialInd==1
				SumAwaMat=AwaMat;
				if SavePhaseStatus == 1
		 			SumRayMat=RayMat;
				end
			else 	
				SumAwaMat=SumAwaMat+AwaMat;
				if SavePhaseStatus == 1
					SumRayMat=SumRayMat+RayMat;
				end
			end
			if SurfTrialStatus
				if TrialInd==1
					hFig=figure('Units','normal','NumberTitle','off','Position',[0 0 .5 .5],'Name','Trial 1');
				else
					figure(hFig);
					set(hFig,'Name',['Trial ',sprintf('%g',TrialInd)]);
				end
				MeanMat=squeeze(mean(AwaMat,1))';
				MinAmp=min(min(MeanMat));
				MaxAmp=max(max(MeanMat));
				fprintf('\n\n');
				fprintf('Min. Amplitude: %g\n\n',MinAmp);
				fprintf('Max. Amplitude: %g\n\n',MaxAmp);
				surf(MeanMat)
                axis([1,NPoints,1,NF0Vec,MinAmp,MaxAmp])
				view(0,90)
				shading interp
				XTickVec=get(gca,'xtick');
				XTickLabelVec=XTickVec.*1000./SampRate;
				set(gca,'xtick',XTickVec,'XTickLabel',XTickLabelVec);
				YTickVec=get(gca,'ytick');
 				YTickLabelVec=F0FrequVec(YTickVec);
				set(gca,'ytick',YTickVec,'YTickLabel',YTickLabelVec);
				xlabel('Time [ms]:')
				ylabel('Frequency [Hz]:')
				title(['Min. Amp: ',sprintf('%g',MinAmp),';  Max. Amp.: ',sprintf('%g',MaxAmp)])
				pause(.01)	
				% 					hFigChild=get(hFig,'children');
				% 					hFigChildChild=get(hFigChild,'children');
				% 				else
				% 					set(hFig,'Name',['Trial ',sprintf('%g',TrialInd)]);
				% 					set(hFigChildChild,'ZData',MeanMat);
				% 					set(hFigChildChild,'SpecularColorReflectance',1)
				% 					pause(.01)
				% 				end
			end
		end
		if NStore ~= 1 
			fprintf(1,'Resize Matrix:\n\n')
			SumAwaMat=SumAwaMat(:,[1:NStore:NPoints],:);
			if SavePhaseStatus == 1
				SumRayMat=SumRayMat(:,[1:NStore:NPoints],:);
			end
		end;
		SumAwaMat= SumAwaMat./NTrials;
		if SavePhaseStatus == 1
			PhaseMat=abs(SumRayMat./NTrials);
		end
		[AwaFile,AwaPath,AwaFilePath]=GetFileNameOfMat(AwaFileMat,FileInd);
		SaveWaveletData(SumAwaMat,AwaFilePath,[],[],[],SampRate,chAvgRef,StartF0,EndF0,DeltaF0,NTap,SigmaM,EegMegStatus,NChanExtra,1,NTrials);
		if SavePhaseStatus == 1
			[PlfFile,PlfPath,PlfFilePath]=GetFileNameOfMat(PwaFileMat,FileInd);
			SaveWaveletData(PhaseMat,PlfFilePath,[],[],[],SampRate,chAvgRef,StartF0,EndF0,DeltaF0,NTap,SigmaM,EegMegStatus,NChanExtra,2,NTrials);
		end
	end
	Message=char('');
	Message=char(Message,'');
	Message=char(Message,'Wavelet calculation completed.');
	if strcmp(computer,'SOL2') | strcmp(computer,'PCWIN')
		set(hChild2ProgressMsgbox,'String',Message);
	else
		set(hChildProgressMsgbox(2),'String',Message);
	end
	set(hChildProgressMsgbox(length(hChildProgressMsgbox)),'Visible','on');
else
  	disp(sprintf('WaveApp: action string ''%s'' not recognized, no action taken.',action))
end	
return;
