function lyngby_batch_fg(filenames, filenameRun, varargin)

% lyngby_batch_fg    - Batch for Ardekani's F-test with nuisance
%
%       lyngby_batch_fg(filenames, 'PropertyName', 'PropertyValue')
%     
%       Input:    filenames   ANALYZE filenames as cell strings
%
%       Property: Path        { {''} ] filepath for result files
%                 Prefix      [ {lyngby} | a string ] Filename
%                             prefix for the result files.
%
%       Batch Ardekani's F-test with nuisance subspace.
%
%       See also LYNGBY, LYNGBY_BAF2_MAIN, LYNGBY_READ_VOLUMES,
%                LYNGBY_WRITE_VOLUMES. 
%
% $Id: lyngby_batch_baf2.m,v 1.1 2003/05/26 17:33:07 fnielsen Exp $


    % Check input arguments
    if nargin < 1
      error('Too few input arguments');
    end
    if ~iscellstr(filenames)
      error('Input argument for ''filenames'' should be cellstrings');
    end
    if mod(length(varargin), 2)
      error([ 'Properties should contain an equal number arguments. ' ...
	    'Both property name and property value'])
    end
    
    
    % Default values
    datatype = 'int16';
    filename_path   = '';
    filename_prefix = 'lyngby';
    components      = 8;
    errorlevel      = 0.001;
    maxNuisanceDim  = 10;
    Vmask           = [];
    
    
    % Read properties
    n = 1;
    while n < length(varargin)
      arg = lower(varargin{n}); 

      if strcmp(arg, 'path')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_path = arg;
	else
	  error('The argument with ''Path'' should be a string');
	end

      elseif strcmp(arg, 'prefix')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_prefix = arg;
	else
	  error('The argument with ''Prefix'' should be a string');
	end

      elseif strcmp(arg, 'volmask') 
	n = n + 1;
	arg = varargin{n};
	if isstruct(arg) 
	  Vmask = arg;
	else
	  error('Argument with ''VolMask'' should be a structure');
	end
	
      else
	error(sprintf('Invalid property: %s', arg));
      end
      n = n + 1;
    end
    
    
    % Filenames of result
    filename_prefix   = fullfile(filename_path, filename_prefix);
    
    
    % Read data
    X = lyngby_read_volumes(filenames, 'VolMask', Vmask);

    
    % Load paradigm
    S = load(filenameRun);
    if isnumeric(S)
      R = S(:);
    else
      error('Could not read run file');
    end
    
    
    % Process
    R.matrix(1,:) = lyngby_baf2_main(X.matrix, R, 'components', ...
	components, 'MaxNuisanceDim', maxNuisanceDim);
    R.rows{1}     = [ filename_prefix '_baf2_f' ];

    
    % Write results
    lyngby_write_volumes(R, 'VolMask', Vmask, 'datatype', datatype);













