function lyngby_batch_cc(filenames, filenameParadigm, varargin)

% lyngby_batch_cc    - Batch for cross-correlation
%
%       lyngby_batch_cons(filenames, 'PropertyName', 'PropertyValue')
%     
%       Input:    Filenames        ANALYZE filenames as cell strings
%                 FilenameParadigm Filename as string of file with
%                                  paradigm                    
% 
%       Property: FilenameVolMask  Filename of volmask
%                 Path             { {''} ] filepath for result files 
%                 Prefix           [ {lyngby} | a string ] Filename
%                                  prefix for the result files.
%                 VolMask          Voxel mask
%
%       Function for batch with cross correlation. 
%
%       See also LYNGBY, LYNGBY_XCORR, LYNGBY_READ_VOLUMES,
%                LYNGBY_WRITE_VOLUMES 
%                      
% $Id: lyngby_batch_cc.m,v 1.2 2004/03/17 19:50:33 fnielsen Exp $


    % Check input arguments
    if nargin < 2
      error('Too few input arguments');
    end
    if ~iscellstr(filenames)
      error('Input argument for ''filenames'' should be cellstrings');
    end
    if mod(length(varargin), 2)
      error([ 'Properties should contain an equal number arguments. ' ...
	    'Both property name and property value'])
    end
    if ~isstr(filenameParadigm)
      error('filenameParadigm should be a string');
    end
    
    
    % Default values
    filename_path   = '';
    filename_prefix = 'lyngby';
    filenameVolMask = '';
    components      = 20;
    Vmask           = [];
    
    
    % Read properties
    n = 1;
    while n < length(varargin)
      arg = lower(varargin{n}); 

      if strcmp(arg, 'filenamevolmask')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filenameVolMask = arg;
	else
	  error('The argument with ''FilenameVolMask'' should be a string');
	end

      elseif strcmp(arg, 'path')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_path = arg;
	else
	  error('The argument with ''Path'' should be a string');
	end

      elseif strcmp(arg, 'prefix')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_prefix = arg;
	else
	  error('The argument with ''Prefix'' should be a string');
	end

      elseif strcmp(arg, 'volmask') 
	n = n + 1;
	arg = varargin{n};
	if isstruct(arg) 
	  Vmask = arg;
	else
	  error('Argument with ''VolMask'' should be a structure');
	end
	
      else
	error(sprintf('Invalid property: %s', arg));
      end
      n = n + 1;
    end
    
    
    if ~isempty(filenameVolMask)
      Vmask = lyngby_read_analyze(filenameVolMask);
    end
    
    
    % Read data
    X = lyngby_read_volumes(filenames, 'VolMask', Vmask);

    
    % Load paradigm
    S = load(filenameParadigm);
    if isnumeric(S)
      PN = S;
    else
      error('Could not read paradigm file');
    end
    

    % Process
    RESULT_CC = lyngby_xcorr(X.matrix, PN, 'Components', components);

    % Filenames of result
    filename_prefix   = fullfile(filename_path, filename_prefix);
    
    R.matrix(1,:) = max(RESULT_CC);
    R.rows{1}     = [ filename_prefix '_cc_max' ];
    
    R.matrix(2,:) = sum(RESULT_CC.^2);;
    R.rows{2}      = [ filename_prefix '_cc_energy' ];
    
    
    % Write results
    lyngby_write_volumes(R, 'VolMask', Vmask, 'datatype', 'int16');










