function lyngby_batch_cons(filenames, varargin)

% lyngby_batch_cons    - Batch for consensus modeling
%
%       lyngby_batch_cons(filenames, 'PropertyName', 'PropertyValue')
%     
%       Input:    filenames   ANALYZE filenames as cell strings
%
%       Property: Path        { {''} ] filepath for result files
%                 Prefix      [ {lyngby} | a string ] Filename
%                             prefix for the result files.
%                 Scaling     [ {HistEq} | none ]
%
%       Batch consensus modeling.
%
%       The consensus (mean) and divergence (std) is written as
%       ANALYZE lyngby_cons_mean.img and lyngby_cons_std.img,
%       respectively, if the 'prefix' property is not set. 
%
%       Example:
%         f = { 'spmT_0003' 'lyngby_ts_t' }
%         lyngby_batch_cons(f)
%
%         lyngby_batch_cons(f, 'prefix', 'test')
%
%         lyngby_batch_cons(f, 'prefix', 'test', 'path', '/tmp')
%         % Is the same as 
%         lyngby_batch_cons(f, 'prefix', '/tmp/test')
%
%       See also LYNGBY, LYNGBY_CONS_MAIN, LYNGBY_READ_ANALYZE,
%                LYNGBY_WRITE_ANA. 
%
% $Id: lyngby_batch_cons.m,v 1.7 2003/05/26 10:32:59 fnielsen Exp $


    % Check input arguments
    if nargin < 1
      error('Too few input arguments');
    end
    if ~iscellstr(filenames)
      error('Input argument for ''filenames'' should be cellstrings');
    end
    if mod(length(varargin), 2)
      error([ 'Properties should contain an equal number arguments. ' ...
	    'Both property name and property value'])
    end
    
    
    % Default values
    datatype        = 'int16';
    filename_path   = '';
    filename_prefix = 'lyngby';
    scaling         = 'histeq';
    
    
    % Read properties
    n = 1;
    while n < length(varargin)
      arg = lower(varargin{n}); 

      if strcmp(arg, 'path')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_path = arg;
	else
	  error('The argument with ''Path'' should be a string');
	end

      elseif strcmp(arg, 'prefix')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_prefix = arg;
	else
	  error('The argument with ''Prefix'' should be a string');
	end

      elseif strcmp(arg, 'scaling')
	n = n + 1;
	arg = lower(varargin{n});
	if isstr(arg) & ( strcmp(arg, 'histeq') | strcmp(arg, 'none') )
	  scaling = arg;
	else
	  error('The argument with ''Scaling'' should be a string');
	end

	
      else
	error(sprintf('Invalid property: %s', arg));
      end
      n = n + 1;
    end
    
    
    filename_prefix   = fullfile(filename_path, filename_prefix);
    filename_res_mean = [ filename_prefix '_cons_mean' ];
    filename_res_std  = [ filename_prefix '_cons_std' ];
    filename_res_svd1 = [ filename_prefix '_cons_svd1' ];


    % Read data
    siz      = lyngby_read_analyze(filenames{1}, 'output', 'size');
    origin   = lyngby_read_analyze(filenames{1}, 'output', 'origin');
    voxelDim = lyngby_read_analyze(filenames{1}, 'output', 'voxeldim');
    V        = lyngby_read_analyze(filenames{1});
    N = length(filenames);
    
    % Allocate datamatrix
    X = zeros(N, prod(size(V)));

    X(1,:) = V(:)';
    
    for n = 2:N
      v = lyngby_read_analyze(filenames{2});
      if length(v) ~= length(V) 
	error(sprintf('Wrong size of %d''th volume', n));
      end
      X(n,:) = v(:)';
    end

    
    % Process
    R = lyngby_cons_main(X, 'scaling', scaling);
    

    % Write results
    lyngby_write_ana(filename_res_mean, R.mean, siz, ...
	'origin', origin, 'voxeldim', voxelDim, 'datatype', datatype, ...
	'scale', 'on');
    lyngby_log(sprintf('Wrote %s (.img/.hdr)', filename_res_mean));
    lyngby_write_ana(filename_res_std, R.std, siz, ...
	'origin', origin, 'voxeldim', voxelDim, 'datatype', datatype, ...
	'scale', 'on');
    lyngby_log(sprintf('Wrote %s (.img/.hdr)', filename_res_std))




