function lyngby_batch_fir(filenames, filenameParadigm, varargin)

% lyngby_batch_fir    - Batch for singular value decomposition
%
%       lyngby_batch_cons(filenames, 'PropertyName', 'PropertyValue')
%     
%       Input:    filenames   ANALYZE filenames as cell strings
%                 filenameParadigm Filename as string of file with
%                             paradigm  
% 
%       Property: Path        { {''} ] filepath for result files
%                 Prefix      [ {lyngby} | a string ] Filename
%                             prefix for the result files.
%
%       See also LYNGBY, LYNGBY_FIR_MAIN, LYNGBY_READ_VOLUMES,
%                LYNGBY_WRITE_VOLUMES 
%                      
% $Id: lyngby_batch_fir.m,v 1.3 2003/05/26 10:34:19 fnielsen Exp $


    % Check input arguments
    if nargin < 2
      error('Too few input arguments');
    end
    if ~iscellstr(filenames)
      error('Input argument for ''filenames'' should be cellstrings');
    end
    if mod(length(varargin), 2)
      error([ 'Properties should contain an equal number arguments. ' ...
	    'Both property name and property value'])
    end
    if ~isstr(filenameParadigm)
      error('filenameParadigm should be a string');
    end
    
    
    % Default values
    filename_path   = '';
    filename_prefix = 'lyngby';
    order           = 7;
    regmethod       = 'pcr';
    regularization  = 0;
    pcrcomp         = 3;
    convtype        = 'skip';
    Vmask           = [];
    
    
    % Read properties
    n = 1;
    while n < length(varargin)
      arg = lower(varargin{n}); 

      if strcmp(arg, 'path')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_path = arg;
	else
	  error('The argument with ''Path'' should be a string');
	end

      elseif strcmp(arg, 'prefix')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_prefix = arg;
	else
	  error('The argument with ''Prefix'' should be a string');
	end

      elseif strcmp(arg, 'volmask') 
	n = n + 1;
	arg = varargin{n};
	if isstruct(arg) 
	  Vmask = arg;
	else
	  error('Argument with ''VolMask'' should be a structure');
	end
	
      else
	error(sprintf('Invalid property: %s', arg));
      end
      n = n + 1;
    end
    
    
    
    % Read data
    X = lyngby_read_volumes(filenames, 'VolMask', Vmask)

    
    % Load paradigm
    S = load(filenameParadigm);
    if isnumeric(S)
      PN = S;
    else
      error('Could not read paradigm file');
    end
    
    
    % Process
    RESULT_FIR = lyngby_fir_main(PN, X.matrix, ...
        'FilterOrder', order, ...
        'RegMethod', regmethod,...
        'Regularization', regularization,...
        'PCRComp', pcrcomp, ...
        'ConvType', convtype);

    lyngby_log('Computing std ...');
    stdFIRinput = std(PN);

    RESULT_FIR_CONV = lyngby_fir_convolve(PN, RESULT_FIR);

    lyngby_log('Computing STRENGTH_FIR ...');
    stdFIRoutput = std(RESULT_FIR_CONV);

    STRENGTH_FIR = stdFIRoutput ./ (stdFIRinput+realmin);
    RESULT = RESULT_FIR;

    lyngby_log('Computing DELAY_FIR ...');
    [lag,voxels] = size(RESULT_FIR);

    if size(RESULT_FIR,1) == 1
      DELAY_FIR = ones(size(RESULT_FIR));
      SUMCOEF_FIR = RESULT_FIR; 
    else
      DELAY_FIR = sum(RESULT_FIR .* ((1:lag)'*ones(1,voxels))  ./ ...
          (realmin + ones(lag,1) * sum(RESULT_FIR)));
      SUMCOEF_FIR = sum(RESULT_FIR); 
    end
    mask = abs(sum(RESULT_FIR))>0.5*sum(abs(RESULT_FIR));
    DELAY_FIR = DELAY_FIR.*(mask==1)-(mask==0); 


    % Filenames of result
    filename_prefix   = fullfile(filename_path, filename_prefix);
    
    
    R.matrix(1,:) = STRENGTH_FIR;
    R.rows{1}     = [ filename_prefix '_fir_strength' ];
    
    R.matrix(2,:) = sum(RESULT_FIR);
    R.rows{2}      = [ filename_prefix '_fir_sumofcoef' ];
    
    
    % Write results
    lyngby_write_volumes(R, 'VolMask', Vmask, 'datatype', 'int16');











