function lyngby_batch_ica(filenames, filenameParadigm, varargin)

% lyngby_batch_ica    - Batch for independent components analysis
%
%       lyngby_batch_ica(filenames, 'PropertyName', 'PropertyValue')
%     
%       Input:    filenames   ANALYZE filenames as cell strings
%                 filenameParadigm Filename as string of file with
%                             paradigm  
% 
%       Property: Components  [ {10} ] Number of sources
%                 Path        { {''} ] filepath for result files
%                 Prefix      [ {lyngby_ica} | a string ] Filename
%                             prefix for the result files.
%                 Tau         [ {5} ] Time delay parameter
%                 
%       See also LYNGBY, LYNGBY_ICA_MS_EST, LYNGBY_READ_VOLUMES,
%                LYNGBY_WRITE_VOLUMES 
%                      
% $Id: lyngby_batch_ica.m,v 1.4 2004/03/08 22:38:31 fnielsen Exp $


    % Check input arguments
    if nargin < 2
      error('Too few input arguments');
    end
    if ~iscellstr(filenames)
      error('Input argument for ''filenames'' should be cellstrings');
    end
    if mod(length(varargin), 2)
      error([ 'Properties should contain an equal number arguments. ' ...
	    'Both property name and property value'])
    end
    if ~isstr(filenameParadigm)
      error('filenameParadigm should be a string');
    end
    
    
    % Default values
    filename_path   = '';
    filename_prefix = 'lyngby';
    components      = 10;
    tau             = 5;
    Vmask           = [];
    
    
    % Read properties
    n = 1;
    while n < length(varargin)
      arg = lower(varargin{n}); 

      if strcmp(arg, 'components')
	n = n + 1;
	arg = varargin{n};
	if isnumeric(arg) & length(arg)==1 & arg==round(arg) & arg > 0
	  components = arg;
	else
	  error([ 'The argument with ''Components'' should be ' ...
		'a positive integer']);
	end
      
      elseif strcmp(arg, 'path')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_path = arg;
	else
	  error('The argument with ''Path'' should be a string');
	end

      elseif strcmp(arg, 'prefix')
	n = n + 1;
	arg = varargin{n};
	if isstr(arg) 
	  filename_prefix = arg;
	else
	  error('The argument with ''Prefix'' should be a string');
	end

      elseif strcmp(arg, 'tau')
	n = n + 1;
	arg = varargin{n};
	if isnumeric(arg) & length(arg)==1 & arg==round(arg) & arg > 1
	  tau = arg;
	else
	  error([ 'The argument with ''Tau'' should be ' ...
		'a positive integer']);
	end
      
      elseif strcmp(arg, 'volmask') 
	n = n + 1;
	arg = varargin{n};
	if isstruct(arg) 
	  Vmask = arg;
	else
	  error('Argument with ''VolMask'' should be a structure');
	end
	
      else
	error(sprintf('Invalid property: %s', arg));
      end
      n = n + 1;
    end
    
    
    
    % Read data
    X = lyngby_read_volumes(filenames, 'VolMask', Vmask);

    
    % Load paradigm
    S = load(filenameParadigm);
    if isnumeric(S)
      PN = S;
    else
      error('Could not read paradigm file');
    end

    
    % Process
    [RESULT_ICA_TIME, RESULT_ICA_IMAGE] = ...
	lyngby_ica_ms_est(X.matrix, ...
	'Components', components, ...
	'Paradigm', PN, ...
	'Tau', tau);


    % Filenames of result
    filename_prefix   = fullfile(filename_path, filename_prefix);
    

    R.matrix = RESULT_ICA_IMAGE;
    for n = 1:size(RESULT_ICA_IMAGE,1)
      R.rows{n}     = [ filename_prefix sprintf('_ica_image%d',n) ];
    end
      
    
    % Write results
    lyngby_write_volumes(R, 'VolMask', Vmask, 'datatype', 'int16');










