function P = lyngby_cdf_fishersg(g, n, varargin)

% lyngby_cdf_fishersg  - Cumulative distribution for Fisher's g
%
%       P = lyngby_cdf_fishersg(g, n)
% 
%       Input:  g   Value where the function is evaluated
%               n   Number of frequency components
%
%       Output: P   Lower tail area
%
%       Lower tail area in the probability density function of
%       Fisher's g, - a statistics used in harmonic analysis/spectral
%       analysis. 
%
%       n is related to the number of data points by: 
%               N = 2n + 1 (odd N)  or   N = 2n + 2 (even N)
%
%       Example:
%         N = 51;
%         x = randn(1,N);                % Data, N(0,1)
%         f = fft(x);
%         p = abs(f(2:ceil(N/2))).^2;    % Scaled periodogram
%         g = max(p)/sum(p);             % Fisher's g
%         lyngby_cdf_fishersg(g, length(p))
%
%       Ref: Fisher (1929), Proc Roy Soc A 125:54-59.
%            Fisher (1940), Ann of Eugenics, 10:14-17 
%
%       See also LYNGBY, LYNGBY_CDF_BETA, LYNGBY_CDF_CHI2,
%                LYNGBY_CDF_GAUSS, LYNGBY_CDF_T, LYNGBY_CDF_WILRANK.  
%
% $Id: lyngby_cdf_fishersg.m,v 1.1 2003/02/11 17:15:19 fnielsen Exp $


    % Check input argument
    if nargin < 2
      error('Too few input arguments')
    end

    
    % M is the greatest integer less than 1/g
    K = floor(min(1/g, n));

    P = 0;
    for k = 1:K
      P = P + (-1)^(k-1) * nchoosek(n,k) * (1-k*g)^(n-1);
    end

    P = 1 - P; 





