function P = lyngby_cdf_wilrank(X, n, recursive)

% lyngby_cdf_wilrank   - "Wilcoxon sign rank" distribution 
%
%       function P = lyngby_cdf_wilrank(X,n)
%
%       Input:  X   Evaluation point
%               n   Number of samples
%
%       Output: P   Distribution function
%
%       Distribution in connection with the Wilcoxon sign rank
%       statistics.
%
%       Ref: Owen, Handbook of statistical tables, 1962
%
%       Example: 
%         lyngby_cdf_wilrank(5, 9)
%
%       See also LYNGBY, LYNGBY_TEST_WILRANK, LYNGBY_CDF_BETA,
%                LYNGBY_CDF_CHI2, LYNGBY_CDF_F, LYNGBY_CDF_FISHERSG,
%                LYNGBY_CDF_GAUSS, LYNGBY_CDF_T.
%
% $Id: lyngby_cdf_wilrank.m,v 1.3 2003/08/07 16:18:03 fnielsen Exp $



    x = X;

    % Cumulated frequencies
    if x == 0
      F = 1;
    elseif x < 0
      F = 0;
    elseif x > n*(n+1)/2
      % Symmetric distribution
      F = lyngby_cdf_wilrank(n*(n+1)/2, n, 1); 
    else
      F = lyngby_cdf_wilrank(x, n-1, 1) + ...
	  lyngby_cdf_wilrank(x-n, n-1, 1);
    end

    % Convert frequencies to probability
    if nargin == 2
      P = F / 2^n;
    else
      P = F;
    end
