function [siz, vdim, name, datatype] = lyngby_getinfo(index);

% lyngby_getinfo       - Get volume info from a file
%
%       function [siz, vdim, name, datatype] = lyngby_getinfo(index);
%
%       Input:  index      Volume index, indexed from one
%
%       Output: siz        Number of voxels in each dimension
%               vdim       Size of a single voxel, in cm's
%               name       A name for the data
%               datatype   Fread style type specifying string (optional!)
%
%       Tries to get the size, voxel dim, and data type information
%       from the file 
%       This information may be obtained by reading the header, if
%       standard or custom data formats are selected. For raw data
%       format, only the byte size of the file can be reported (in siz)
%
%       For the custom file format, the user should supply a function
%
%         data_readinfo.m
%
%       defined as 
%              
%         function [siz, vdim, name, datatype] = data_readinfo(index)
%
%       See also LYNGBY, LYNGBY_GETVOLUME.
%
% $Id: lyngby_getinfo.m,v 1.9 2002/03/21 13:51:08 fnielsen Exp $ 


    lyngby_global;
	
    if FILE_READING_TYPE == 1,
      % Analyze/VAPET format
      name = [FILENAME_PATH ...
	    sprintf(FILENAME_PATTERN, index+FILENAME_STARTINDEX-1)];
      [siz, vdim, datatype] = lyngby_read_header(name);
      
    elseif FILE_READING_TYPE == 2,
      % Custom format
      if exist('data_readinfo')
	% Call to data_readinfo hidden in an eval 
	% Else the syntax checker/ function call checker is complaining
	eval('[siz, vdim, name, datatype] = data_readinfo(index);');
      else
	error('The user should provide a ''data_readinfo'' function.');
      end
      
    elseif FILE_READING_TYPE == 3,
      % RAW format
      name = [FILENAME_PATH ...
	    sprintf(FILENAME_PATTERN, index+FILENAME_STARTINDEX-1)];
      f = fopen(name, 'rb');
      if f<0, 
	error(sprintf('Unable to open file: ''%s''', name));
      else
	fseek(f, 0, 1);
	bytes = ftell(f);  % Return file size in bytes 
	datatype = DATATYPE;
	vdim = VOXELDIM;
	if strcmp(datatype, 'short')
	  bpv = 2;
	elseif strcmp(datatype, 'int')
	  bpv = 4;
	elseif strcmp(datatype, 'long')
	  bpv = 8;
	elseif strcmp(datatype, 'ushort')
	  bpv = 2;
	elseif strcmp(datatype, 'uint')
	  bpv = 4;
	elseif strcmp(datatype, 'ulong')
	  bpv = 8;
	elseif strcmp(datatype, 'float')
	  bpv = 4;
	elseif strcmp(datatype, 'double')
	  bpv = 8;
	elseif strcmp(datatype, 'byte')
	  bpv = 1;
	else
	  error(sprintf('Unknown datatype: %s', datatype));
	end
	z = bytes / prod(NUM_VOXELS(1:2)) / bpv;
	if z ~= round(z)
	  error(sprintf(['Number of bytes in the file does not fit '...
		'the number of voxels and datatype.\n' ...
		'Read: %d; x: %d; y: %d; byte per voxel: %d; ' ...
		'thus z: %d '], ...
	      bytes, NUM_VOXELS(1), NUM_VOXELS(2), bpv, z)); 
	else
	  siz = [NUM_VOXELS(1) NUM_VOXELS(2) z];
	end
      end
    else
      error('Invalid FILE_READING_TYPE');
    end

