function Y = lyngby_pois_forwn(X, Poisson, kernelWidth, ...
    convType)

% lyngby_pois_forwn    - Possion kernel forward (prediction)
%
%      function [Y] = lyngby_pois_forwn(X, Poisson,
%          kernelWidth, convType)
%
%      Input:  X             Input to the linear system
%              Poisson       Poisson parameter
%              kernelWidth   Kernel Width (optional, 
%                            default min([size(X,1) 2*max(Poisson)]) )
%              convType      Convolution type: What should be done
%                            with the first samples:
%                            1: Skip the number of samples
%                            corresponding to the kernel width.
%                            2: Fit all and use zeropadding
%                            3: Fit all and use pad with the first
%                            sample
%                            4: Wrap the time-series 
%                            (optional, default 1)
%
%      Output: Y             Prediction, the input forward (convolved)
%                            with the kernel.
%
%      See also LYNGBY, LYNGBY_POIS_ERROR LYNGBY_POIS_FORWARD,
%               LYNGBY_POIS_MAIN. 
%
% $Id: lyngby_pois_forwn.m,v 1.3 2003/05/23 09:45:57 fnielsen Exp $


    [rX, cX]             = size(X);
    [rPoisson, cPoisson] = size(Poisson);

    % Default parameters
    if nargin < 4
      convType = 1;
      if nargin < 3
	kernelWidth = min([rX ceil(2*max(Poisson))]);
      end
    end

    if kernelWidth <= 0 kernelWidth > rX
      error('kernelWidth should be: 0 <= kernelWidth < rows of x');
    end

    if convType == 1
      % Skip the first samples

      nFFT = 2^nextpow2(rX);
      H = lyngby_pdf_poisson((0:kernelWidth-1)', Poisson);
      if length(H) == 1
	Y = real(ifft( fft(X,nFFT) * ones(1, size(H,2)) .* ...
	    fft(H,nFFT)'));
      else
	Y = real(ifft( fft(X,nFFT) * ones(1, size(H,2)) .* ...
	    fft(H,nFFT)));
      end
      Y = Y(kernelWidth:rX,:);
      
    elseif convType == 2
      % Zeropad

      nFFT = 2^nextpow2(rX);
      H = lyngby_pdf_poisson((0:kernelWidth-1)', Poisson);
      if length(H) == 1
	Y = real(ifft( fft(X,nFFT)  * ones(1, size(H,2)) .* fft(H,nFFT)'));
      else
	Y = real(ifft( fft(X,nFFT)  * ones(1, size(H,2)) .* fft(H,nFFT)));
      end
      Y = Y(1:rX,:);
      
    elseif convType == 3
      % Extend
      
      H = lyngby_pdf_poisson((0:kernelWidth-1)', Poisson);
      rXpad = rX + kernelWidth - 1;
      nFFT = 2^nextpow2(rXpad);
      if length(H) == 1
	Y = real(ifft(fft(X([ones(1,kernelWidth) 2:rX],:), nFFT) * ...
	    ones(1, size(H,2)) .* fft(H, nFFT)'));
      else
	Y = real(ifft(fft(X([ones(1,kernelWidth) 2:rX],:), nFFT) * ...
	    ones(1, size(H,2)) .* fft(H, nFFT)));
      end
      Y = Y(kernelWidth:(rX+kernelWidth-1), :);
      
    elseif convType == 4
      % Wrap
      
      H = lyngby_pdf_poisson((0:kernelWidth-1)', Poisson);
      if length(H) == 1
	Y = real(ifft(fft(X) * ones(1, size(H,2)) .* fft(H, rX)'));
      else
	Y = real(ifft(fft(X) * ones(1, size(H,2)) .* fft(H, rX)));
      end
      
    end





















