function [Prob,T] = lyngby_ts_main(P, X, arg1, arg2, arg3, arg4)

% lyngby_ts_main       - Whole volume Student's t-test for equal variances.
%
%       function [Prob,T] = lyngby_ts_main(P, X, 'PropertyName',
%          'PropertyValue') 
%      
%       Input:    P         Paradigm
%                 X         Datamatrix
%
%       Property: EdgeDrop  The number of scans to discard at the
%                           transitions of the paradigm
%
%       Output:   Prob      Probability (p-value)
%                 T         Student's unpaired t-value
%
%       This function gives the probability that Student's t
%       calculated on data X1 and X2, sampled from distributions
%       with the same variance, is higher than observed, i.e.
%       the "significance" level. This is used to test whether
%       two sample have significantly different means.
%       The whole volume is tested (compare with lyngby_ts_ttest)
%       and X1 are taken from the on-periods and X2 from the off-periods.
%       A number of scans (edgedrop) can be left out of the analysis
%       in order to accommodate to the heamodynamical response time.
%
%       The smaller P is, the more significant the difference between
%       the means. E.g. if P = 0.05 or 0.01, it is very likely that
%       the two sets are sampled from distributions with different
%       means.
%
%       This assumes that the samples are drawn from distributions with
%       the SAME VARIANCE. The test is unpaired.
%
%       See also: LYNGBY, LYNGBY_TS_TTEST, LYNGBY_TS_UTTEST,
%                 LYNGBY_TS_PTEST, LYNGBY_BAT_TEST.
%
% $Id: lyngby_ts_main.m,v 1.10 2002/11/18 17:48:01 fnielsen Exp $


    % Check arguments
    [rX, cX] = size(X);
    [rP, cP] = size(P);
    if rX ~= rP
      error(sprintf([ 'The datamatrix X and the paradigm ' ...
	    'has different number of rows.\n X: %dx%d\n P: %dx%d'], ...
	    size(X), size(P)));
    end
    if cP ~= 1
      error(sprintf(['The paradigm P should only contain one ' ...
	    'column.\n P: %dx%d'], size(P)));
    end
    
    
    % Default properties
    edgedrop = 0;
    
  
    % Parse Properties
    n = 1;
    while n <= nargin-2
      eval(sprintf('arg = lower(arg%d);', n));

      if strcmp(arg, 'edgedrop')
	n = n + 1;
	eval(sprintf('arg = lower(arg%d);', n));
	if isreal(arg) & arg == round(arg) & arg >= 0 
	  edgedrop = arg;
	else
	  error('The argument with ''Edgedrop'' should be a positive integer.'); 
	end
	
      else
	error(sprintf('Invalid property: %s', arg));
      end
      n = n + 1;
    end
    
    
    % Allocate    
    Prob = zeros(1, cX);
    T = zeros(1, cX);

    % Get edgedrop mask
    [ind1,ind2] = lyngby_splitonoff(P, edgedrop);

    % Iterate over voxels
    for i = 1:cX
      x = X(:,i);
      if lyngby_mod(i, 100) == 0
	lyngby_log(sprintf('Doing t-test: %d / %d', i, cX));
      end
      if any(x)
	[Prob(i),T(i)] = lyngby_ts_ttest(x(ind1), x(ind2));
      else
      	Prob(i) = 1;
	T(i) = 0;
      end
    end

















