function lyngby_ui_view_cons(varargin)

% lyngby_ui_view_cons   - Consensus viewing function
%
%       function lyngby_ui_view_cons(PropertyName, PropertyValue)
%
%       Property: type       [ {time} | time, original | histogram |
%                            Mean sequence | Std sequence |
%                            Mean & Std Sequence |
%                            Mean/std | Mean/Std, voxel | eigenspace ]
%                 voxel      Voxel to be plotted
%                 index      Time indices to be shown
%                 amplitude  [ {free} | keep ] axis control
%                 value      Used in the title
% 
%       This functions is automatically called from lyngby_ui_view. It
%       plots time series associated with consensus modeling. Some of
%       the data is read from global variables. 
% 
%       See also LYNGBY, LYNGBY_UI_VIEW, LYNGBY_UIS_CONS_V,
%                LYNGBY_UIS_CONS, LYNGBY_CONS_MS_EST, LYNGBY_UI_GLOBAL.
%
% $Id: lyngby_ui_view_cons.m,v 1.1 2002/03/12 16:06:35 fnielsen Exp $      


    lyngby_global
    lyngby_ui_global

    % Default values
    plotparam = [ 1 2]; 
    plottype  = 'time';
    voxel     = 1;
    indices   = 1:size(XN, 1);
    amplitude = 1;
    value     = NaN;
    
    n = 1;
    while n < nargin
      arg = lower(varargin{n}); 
      if strcmp(arg, 'type')
	n = n + 1;
	arg = deblank(lower(varargin{n}));
	if isstr(arg)
	  if strcmp(arg, 'time') | ...
		strcmp(arg, 'time, original') | ...
		strcmp(arg, 'histogram') | ...
		strcmp(arg, 'mean sequence') | ...
		strcmp(arg, 'std sequence') | ...
		strcmp(arg, 'mean & std sequence') | ...
		strcmp(arg, 'mean/std') | ...
		strcmp(arg, 'mean/std, voxel') 
	    plottype = arg;
	  elseif ~isempty(findstr(arg, 'eigenspace'))
	    plottype = 'eigenspace';
	  else
	    error(sprintf('Argument to ''type'' not recognized: %s', arg));
	  end
	else	  
	  error('''type'' PropertyName should be followed by a string');
	end
      elseif strcmp(arg, 'voxel')
	n = n + 1;
	arg = varargin{n};
	if length(voxel) == 1
	  voxel = arg;
	else
	  error(['''voxel'' PropertyName should be followed by a ' ...
	      'a single value']);
	end
      elseif strcmp(arg, 'index')
	n = n + 1;
	arg = varargin{n};
	if length(arg) == 2
	  indices = arg(1):arg(2);
	else
	  error('')
	end

      elseif strcmp(arg, 'amplitude')
	n = n + 1;
	arg = lower(varargin{n});
	if strcmp(arg, 'free')
	  amplitude = 1;
	elseif strcmp(arg, 'keep')
	  amplitude = 2;
	else
	  % disp('Warning: ''Amplitude'' PropertyName not recognized');
	end
      elseif strcmp(arg, 'value')
	n = n + 1;
	arg = varargin{n};
	value = arg;
      else
	error(sprintf('Invalid property: %s', arg));
      end
      n = n + 1;
    end

    % 'keep' only if same type as old plot
    previousPlottype = '';
    h = findobj(gcf, 'tag', 'previousPlottype');
    if h
      previousPlottype = get(h, 'Userdata');
    end
    if isempty(previousPlottype) | ~strcmp(plottype, previousPlottype)
      amplitude = 1;
    end


    oldAxis = axis;
    
    if ~isempty(voxel)
      voxel3 = lyngby_mask2full(voxel);

      if strcmp(plottype, 'time')
	% Time series plot with XN
	
	xn = XN(:,voxel);
	
	time = 1:length(xn);
	plot(time, xn, 'g-');
	
	if ~isempty(PN) & any(PN);
	  scal = max(abs(xn))/max(abs(PN));
	  hold on
	  plot(time, 0.95*scal*PN, 'b:');
	  legend('Data', 'Paradigm');
	  hold off;
	end

	sTitle = 'Time series';

      elseif strcmp(plottype, 'time, original')
	% Time series plot, with X
	
	x = X(:,voxel);
	
	time = 1:length(x);
	plot(time, x, 'g-');
	
	if ~isempty(PN);
	  scal = max(abs(x))/max(abs(PN));
	  hold on
	  plot(time, 0.95*scal*PN, 'b:');
	  legend('Data', 'Paradigm');
	  hold off;
	end

	sTitle = 'Original time series';

	
      elseif strcmp(plottype, 'histogram')
	% Histogram
	
	xn = XN(:,voxel);
	hist(xn,25);
	sTitle = 'Histogram of data';
      
      elseif strcmp(plottype, 'mean sequence')
	% Mean Sequence
	
	plot(X_SEQMEAN)
	xlabel('Scan');
	ylabel('Mean of image');
	sTitle = 'Mean sequence';
      
      elseif strcmp(plottype, 'std sequence')
	% Std Sequence
	
	plot(X_SEQSTD)
	xlabel('Scan');
	ylabel('Standard deviation of image');
	sTitle = 'Std Sequence';
      
      elseif strcmp(plottype, 'mean & std sequence')
	% Mean Sequence
	
	plot(1:length(X_SEQMEAN), X_SEQMEAN, '-', ...
	    1:length(X_SEQSTD),  X_SEQMEAN + X_SEQSTD, ':', ...
	    1:length(X_SEQSTD),  X_SEQMEAN - X_SEQSTD, ':')
	xlabel('Scan');
	ylabel('Mean & Standard deviation of image');
	sTitle = 'Mean & Std Sequence';
      
      elseif strcmp(plottype, 'mean/std')
	% mean/std
	
	plot(X_MEAN, X_STD, '.')
	xlabel('Mean of each voxel');
	ylabel('Standard deviation of each voxel');
	sTitle = 'std/mean';
	
	axis([min(X_MEAN) max(X_MEAN) 0 max(X_STD)])
	
      elseif strcmp(plottype, 'mean/std, voxel')
	% mean/std, voxel
	
	plot(X_MEAN, X_STD, '.')
	hold on
	plot(X_MEAN(voxel), X_STD(voxel), 'co', 'Markersize', 12, ...
	    'LineWidth', 5);
	plot(X_MEAN(voxel), X_STD(voxel), 'rx', 'Markersize', 20);
	hold off
	xlabel('Mean of each voxel');
	ylabel('Standard deviation of each voxel');
	sTitle = 'std/mean';

	axis([min(X_MEAN) max(X_MEAN) 0 max(X_STD)])
	
      elseif strcmp(plottype, 'eigenspace')

	comp = plotparam;

	x = RESULT_CONS.svdU(:, comp(1));
	y = RESULT_CONS.svdU(:, comp(2));
	plot(x, y, 'bs');
	for n = 1:length(RESULT_CONS.labels)
	  label = RESULT_CONS.labels{n};
	  text(x(n), y(n), label, 'interpreter', 'none');
	end

	sTitle = sprintf('Eigenspace', comp);
      
      else
	error('Wrong plottype')
      end

      if amplitude == 2
	ax = oldAxis;
      else
	ax = axis;
      end
      if strcmp(plottype, 'time') | strcmp(plottype, 'time, original')
	ax(1:2) = [min(indices) max(indices)];
      end
      axis(ax);

      sTitle = sprintf('%s, [%d, %d, %d]', sTitle, voxel3);
      sTitle = sprintf('%s (%d)', sTitle, voxel);
      if ~isnan(value)
	sTitle = sprintf('%s, %f', sTitle, value);
      end
      title(sTitle);

      
    end

    h = findobj(gcf, 'tag', 'previousPlottype');
    if isempty(h)
      uicontrol(gcf, 'Tag', 'previousPlottype', 'Visible', 'off');
    end
    set(findobj(gcf, 'Tag', 'previousPlottype'), 'UserData', plottype);








